package org.simantics.modeling.ui.diagramEditor;

import java.util.ArrayDeque;
import java.util.Iterator;

import org.eclipse.swt.widgets.Display;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gnu.trove.map.hash.TObjectLongHashMap;


public class DisposingPolicy {

    private static final Logger LOGGER = LoggerFactory.getLogger(DisposingPolicy.class);

    public static final int MAX_QUEUE_LENGTH = 8;
    public static final long DISPOSE_TIME = 30000L; // ms
    public static final long MIN_DELAY = 200L; // ms

    private volatile int maxQueueLength;
    private volatile boolean disposeOverTime;
    private volatile long disposeWaitTimeMs;
    private ArrayDeque<Runnable> disposerQueue;
    private TObjectLongHashMap<Runnable> disposeTime;
    private Runnable currentlyScheduled = null;

    private Runnable disposeOne = () -> {
        // If disposeOverTime is not enabled, then base the actual disposal only on the length of the queue.
        int queueLength = disposerQueue.size();
        boolean debug = LOGGER.isDebugEnabled();
        if(debug)
            LOGGER.debug("disposeOne: queueLength: {}, maxQueueLength: {}", queueLength, maxQueueLength); //$NON-NLS-1$
        if (!disposeOverTime && queueLength <= maxQueueLength)
            return;
        if(queueLength > 0) {
            Runnable runnable = disposerQueue.removeFirst();
            disposeTime.remove(runnable);
            currentlyScheduled = null;
            if(debug)
                LOGGER.debug("Executing disposer {}", runnable); //$NON-NLS-1$
            runnable.run();
            if(debug)
                LOGGER.debug("Executed disposer {}", runnable); //$NON-NLS-1$
            if(!disposerQueue.isEmpty())
                scheduleDispose();
        }
    };

    public DisposingPolicy() {
        this(MAX_QUEUE_LENGTH, true, DISPOSE_TIME);
    }

    public DisposingPolicy(int maxQueueLength) {
        this(maxQueueLength, true, DISPOSE_TIME);
    }

    public DisposingPolicy(int maxQueueLength, boolean disposeOverTime, long disposeWaitTimeMs) {
        this.maxQueueLength = maxQueueLength;
        this.disposeOverTime = disposeOverTime;
        this.disposeWaitTimeMs = disposeWaitTimeMs;
        this.disposerQueue = new ArrayDeque<>(maxQueueLength);
        this.disposeTime = new TObjectLongHashMap<>(maxQueueLength);
    }

    public void reconfigure(int maxQueueLength, boolean disposeOverTime, int disposeWaitTimeMs) {
        this.maxQueueLength = maxQueueLength;
        this.disposeOverTime = disposeOverTime;
        this.disposeWaitTimeMs = disposeWaitTimeMs;
        scheduleDispose();
    }

    public void setMaxQueueLength(int maxQueueLength) {
        this.maxQueueLength = maxQueueLength;
    }

    public void setDisposalTimes(boolean disposeOverTime, long disposeWaitTimeMs) {
        this.disposeOverTime = disposeOverTime;
        this.disposeWaitTimeMs = disposeWaitTimeMs;
        scheduleDispose();
    }

    private void scheduleDispose() {
        boolean debug = LOGGER.isDebugEnabled();
        if(debug)
            LOGGER.debug("scheduleDispose({}, {}, {})", disposeOverTime, disposerQueue.size(), maxQueueLength); //$NON-NLS-1$
        if (!disposeOverTime && disposerQueue.size() <= maxQueueLength)
            return;
        currentlyScheduled = disposerQueue.peekFirst();
        if (currentlyScheduled == null)
            return;
        long delay = Math.max(
                disposeTime.get(currentlyScheduled) - System.currentTimeMillis(),
                MIN_DELAY);
        Display.getCurrent().timerExec((int)delay, disposeOne);
        if(debug)
            LOGGER.debug("Scheduled disposer {} in {} ms", currentlyScheduled, delay); //$NON-NLS-1$
    }

    /**
     * Runs the disposer either after {@link #disposeWaitTimeMs} if
     * {@link #disposeOverTime} is <code>true</code> or when there are more than
     * {@link #maxQueueLength} disposers active and this is first of them (by
     * activation order). This method must be called from UI thread.
     */
    public void addDisposer(Runnable disposer) {
        if(LOGGER.isDebugEnabled())
            LOGGER.debug("Added disposer {}", disposer); //$NON-NLS-1$
        if(disposeTime.contains(disposer))
            return;

        long t = System.currentTimeMillis();
        disposerQueue.addLast(disposer);
        disposeTime.put(disposer, t + disposeWaitTimeMs);

        // Keep the number of alive hidden diagrams under the requested amount by
        // marking the deadline for first disposeCount disposers to be right now.
        int disposeCount = disposerQueue.size() - maxQueueLength;
        if (disposeCount > 0) {
            Iterator<Runnable> it = disposerQueue.iterator();
            for (int i = 0; i < disposeCount && it.hasNext(); ++i) {
                disposeTime.put(it.next(), t);
            }
        }

        scheduleDispose();
    }

    /**
     * Cancels a disposer added before. This method must be called from
     * UI thread.
     */
    public void removeDisposer(Runnable disposer) {
        if(LOGGER.isDebugEnabled())
            LOGGER.debug("Removed disposer {}", disposer); //$NON-NLS-1$
        disposerQueue.remove(disposer);
        disposeTime.remove(disposer);
        if(disposer == currentlyScheduled) {
            currentlyScheduled = null;
            if(!disposerQueue.isEmpty())
                scheduleDispose();
        }
    }

}
