/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.diagramEditor;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Set;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.viewers.CellLabelProvider;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.ICheckStateProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TreeEditor;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.part.Page;
import org.simantics.diagram.layer.ILayersViewPage;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.diagram.DiagramHints;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.diagram.participant.Selection;
import org.simantics.g2d.element.ElementClass;
import org.simantics.g2d.element.ElementHints;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.element.handler.ElementLayerListener;
import org.simantics.g2d.element.handler.ElementLayers;
import org.simantics.g2d.layers.IEditableLayer;
import org.simantics.g2d.layers.ILayer;
import org.simantics.g2d.layers.ILayers;
import org.simantics.g2d.layers.ILayers.ILayersListener;
import org.simantics.g2d.layers.ILayersEditor;
import org.simantics.g2d.layers.SimpleLayer;
import org.simantics.utils.datastructures.Arrays;
import org.simantics.utils.datastructures.disposable.IDisposable;
import org.simantics.utils.datastructures.disposable.IDisposeListener;
import org.simantics.utils.datastructures.hints.HintListenerAdapter;
import org.simantics.utils.datastructures.hints.IHintContext.Key;
import org.simantics.utils.datastructures.hints.IHintListener;
import org.simantics.utils.datastructures.hints.IHintObservable;
import org.simantics.utils.ui.ISelectionUtils;

public class DiagramLayersPage extends Page implements ILayersViewPage {

    private static final String TEXT_IGNORE_FOCUS_SETTINGS = Messages.DiagramLayersPage_FocusAll;
    private static final String TOOLTIP_IGNORE_FOCUS_SETTINGS = Messages.DiagramLayersPage_FocusAllTT;

    private static final String TEXT_IGNORE_VISIBILITY_SETTINGS = Messages.DiagramLayersPage_ShowAll;
    private static final String TOOLTIP_IGNORE_VISIBILITY_SETTINGS = Messages.DiagramLayersPage_ShowAllTT;

    final private ICanvasContext context;
    private CheckboxTreeViewer viewer;
    private Button ignoreVisibilityButton;
    private Button ignoreFocusButton;
    private Composite composite;
    private TreeEditor editor;

    private Collection<IElement> elements = Collections.emptySet();
    private ILayersEditor layers;
    
    enum Attribute {
        Visible,
        Focusable
    }

    enum Tristate {
        True,
        False,
        Both;

        static Tristate to(Boolean b) {
            return b == null ? null : b ? True : False;
        }
        boolean toBoolean() {
            switch (this) {
                case Both:
                    throw new IllegalStateException("cannot convert Tristate Both to boolean"); //$NON-NLS-1$
                case False:
                    return false;
                case True:
                    return true;
                default:
                    return false;
            }
        }
        Tristate toggle() {
            switch (this) {
                case Both:
                case False:
                    return True;
                case True:
                    return False;
                default:
                    return False;
            }
        }
        Tristate merge(Tristate state) {
            if (state == null)
                return this;
            switch (this) {
                case Both:
                    return Both;
                case False:
                    switch (state) {
                        case False:
                            return False;
                        case Both:
                        case True:
                            return Both;
                    }
                case True:
                    switch (state) {
                        case True:
                            return True;
                        case False:
                        case Both:
                            return Both;
                    }
            }
            return this;
        }
    }

    Boolean getAttribute(IElement e, ILayer layer, Attribute attribute) {
        ElementClass ec = e.getElementClass();
        for (ElementLayers el : ec.getItemsByClass(ElementLayers.class)) {
            switch (attribute) {
                case Visible:
                    return Boolean.valueOf(el.isVisible(e, layer));
                case Focusable:
                    return Boolean.valueOf(el.isFocusable(e, layer));
            }
        }
        return null;
    }

    Tristate getJointAttribute(Collection<IElement> elements, ILayer layer, Attribute attribute) {
        Tristate state = null;
        for (IElement e : elements) {
            Boolean attr = getAttribute(e, layer, attribute);
            if (attr == null)
                continue;

            if (state == null) {
                state = Tristate.to(attr);
            } else {
                state = state.merge(Tristate.to(attr));
            }
        }
        return state;
    }

    void setAttribute(Collection<IElement> elements, ILayer layer, Attribute attribute, boolean value) {
        // Short-circuit the hint updates so that the result will be immediately available for viewer.refresh().
        // There is no listener for the element layer hints so the viewer won't update the tristates automatically!

        IDiagram diagram = context.getDefaultHintContext().getHint(DiagramHints.KEY_DIAGRAM);
        ElementLayerListener elementLayerListener = diagram.getHint(DiagramHints.KEY_ELEMENT_LAYER_LISTENER);
        switch (attribute) {
            case Visible:
                for (IElement e : elements) {
                    Set<ILayer> elementLayers = (Set<ILayer>) e.getHint(ElementHints.KEY_VISIBLE_LAYERS);
                    if (elementLayers != null) {
                        if (value)
                            elementLayers.add(layer);
                        else
                            elementLayers.remove(layer);
                    }
                    elementLayerListener.visibilityChanged(e, layer, value);
                }
                break;
            case Focusable:
                for (IElement e : elements) {
                    Set<ILayer> elementLayers = (Set<ILayer>) e.getHint(ElementHints.KEY_FOCUS_LAYERS);
                    if (elementLayers != null) {
                        if (value)
                            elementLayers.add(layer);
                        else
                            elementLayers.remove(layer);
                    }
                    elementLayerListener.focusabilityChanged(e, layer, value);
                }
                break;
        }
        elementLayerListener.flush();
    }


    final private IHintListener selectionListener = new HintListenerAdapter() {

        @Override
        public void hintChanged(IHintObservable sender, Key key, Object oldValue, Object newValue) {
            Collection<IElement> es = Collections.emptySet();
            if (newValue instanceof Collection<?>) {
                Collection<?> coll = (Collection<?>)newValue;
                es = new ArrayList<IElement>(coll.size());
                for (Object o : coll) {
                    if (!(o instanceof IElement))
                        return;
                    es.add((IElement) o);
                }
                if (es.isEmpty())
                    es = Collections.emptySet();
            }
            elements = es;
            redraw();
        }

        private void redraw() {
            if (viewer != null) {
                viewer.getControl().getDisplay().asyncExec(new Runnable() {
                    @Override
                    public void run() {
                        if (viewer.getControl().isDisposed())
                            return;
                        viewer.getControl().redraw();
                    }
                });
            }
        }
    };
    
    final private IDisposeListener contextDisposeListener = new IDisposeListener() {

        @Override
        public void onDisposed(IDisposable sender) {
            if (getControl() != null) getControl().getDisplay().asyncExec(new Runnable() {
                @Override
                public void run() {
                    dispose();
                }
            });
        }
    };

    public DiagramLayersPage(ICanvasContext context) {
        this.context = context;

        context.getDefaultHintContext().addKeyHintListener(Selection.SELECTION0, selectionListener);
        context.addDisposeListener(contextDisposeListener);
    }

    @Override
    public void dispose() {

        context.getDefaultHintContext().removeKeyHintListener(Selection.SELECTION0, selectionListener);
        context.removeDisposeListener(contextDisposeListener);
        if (layers != null && layersListener != null) {
            layers.removeLayersListener(layersListener);
            layersListener = null;
        }
        super.dispose();

    }

    @Override
    public void createControl(Composite parent) {
        composite = new Composite(parent, SWT.NONE);
        GridLayoutFactory.fillDefaults().numColumns(4).applyTo(composite);
        
        IDiagram diagram = context.getDefaultHintContext().getHint(DiagramHints.KEY_DIAGRAM);
        if (diagram != null) onDiagramSet(diagram);
        context.getDefaultHintContext().addKeyHintListener(DiagramHints.KEY_DIAGRAM, new IHintListener() {
            @Override
            public void hintChanged(IHintObservable sender, Key key, Object oldValue, Object newValue) {
                IDiagram diagram = (IDiagram)newValue;
                onDiagramSet(diagram);
            }

            @Override
            public void hintRemoved(IHintObservable sender, Key key, Object oldValue) {
            }

        });
    }
    
    private void onDiagramSet(IDiagram diagram) {
        if (diagram != null) {
            layers = diagram.getHint(DiagramHints.KEY_LAYERS_EDITOR);
            if (layers != null) initialize(layers, diagram);
            diagram.addKeyHintListener(DiagramHints.KEY_LAYERS_EDITOR, new IHintListener() {
                
                
                @Override
                public void hintChanged(IHintObservable sender, Key key, Object oldValue, Object newValue) {
                    if (newValue != null) {
                        initialize(layers, diagram);
                    }
                }
                @Override
                public void hintRemoved(IHintObservable sender, Key key, Object oldValue) {
                }

            });
        }
    }

    private void initialize(ILayersEditor layers, IDiagram diagram) {
    	composite.getDisplay().asyncExec(new Runnable() {
			@Override
			public void run() {
				initialize2(layers, diagram);
			}
    	});
    }

    private ILayersListener layersListener = new ILayersListener() {

        @Override
        public void changed() {
            scheduleRefresh();
        }

        void scheduleRefresh() {
            viewer.getControl().getDisplay().asyncExec(new Runnable() {
                @Override
                public void run() {
                    viewer.refresh();
                    ignoreVisibilityButton.setSelection(layers.getIgnoreVisibilitySettings());
                    ignoreFocusButton.setSelection(layers.getIgnoreFocusSettings());
                    if (!context.isDisposed())
                        context.getContentContext().setDirty();
                }
            });
        }
    };

    private static String findFreshName(ILayers layers, String proposal) {
        Set<ILayer> all = layers.getLayers();
        String name = proposal;
        int i = 1;
        while (true) {
            boolean match = false;
            for (ILayer layer : all) {
                if (name.equals(layer.getName())) {
                    match = true;
                    break;
                }
            }
            if (!match)
                return name;
            ++i;
            name = proposal + " " + i; //$NON-NLS-1$
        }
    }

    private void initialize2(ILayersEditor layers, IDiagram diagram) {
        layers.addLayersListener(layersListener);

        GridData gridData = new GridData(GridData.FILL, GridData.FILL, true, false);
        composite.setLayoutData(gridData);
        composite.setLayout(new GridLayout(3, false));
        
        ignoreVisibilityButton = new Button(composite, SWT.CHECK);
        ignoreVisibilityButton.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, false, false, 1, 1));
        ignoreVisibilityButton.setText(TEXT_IGNORE_VISIBILITY_SETTINGS);
        ignoreVisibilityButton.setToolTipText(TOOLTIP_IGNORE_VISIBILITY_SETTINGS);
        ignoreVisibilityButton.setSelection(layers.getIgnoreVisibilitySettings());
        ignoreVisibilityButton.addSelectionListener(new SelectionListener() {

            @Override
            public void widgetSelected(SelectionEvent e) {
                layers.setIgnoreVisibilitySettings(!layers.getIgnoreVisibilitySettings());
                context.getThreadAccess().asyncExec(new Runnable() {

                    @Override
                    public void run() {
                        if(context.isDisposed()) return;
                        context.getContentContext().setDirty();
                    }

                });
            }

            @Override
            public void widgetDefaultSelected(SelectionEvent e) {
                widgetSelected(e);
            }

        });

        ignoreFocusButton = new Button(composite, SWT.CHECK);
        ignoreFocusButton.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, false, false, 1, 1));
        ignoreFocusButton.setText(TEXT_IGNORE_FOCUS_SETTINGS);
        ignoreFocusButton.setToolTipText(TOOLTIP_IGNORE_FOCUS_SETTINGS);
        ignoreFocusButton.setSelection(layers.getIgnoreFocusSettings());
        ignoreFocusButton.addSelectionListener(new SelectionListener() {

            @Override
            public void widgetSelected(SelectionEvent e) {
                layers.setIgnoreFocusSettings(!layers.getIgnoreFocusSettings());
                context.getThreadAccess().asyncExec(new Runnable() {

                    @Override
                    public void run() {
                        if(context.isDisposed()) return;
                        context.getContentContext().setDirty();
                    }

                });
            }

            @Override
            public void widgetDefaultSelected(SelectionEvent e) {
                widgetSelected(e);
            }

        });

        Button addButton = new Button(composite, SWT.NONE);
        addButton.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, false, 1, 1));
        addButton.setText(Messages.DiagramLayersPage_New);
        addButton.setToolTipText(Messages.DiagramLayersPage_NewTT);
        addButton.addSelectionListener(new SelectionListener() {

            @Override
            public void widgetSelected(SelectionEvent e) {
                String name = findFreshName(layers, Messages.DiagramLayersPage_NewRole);
                SimpleLayer layer = new SimpleLayer(name);
                layers.addLayer(layer);
                layers.activate(layer);
            }

            @Override
            public void widgetDefaultSelected(SelectionEvent e) {
                widgetSelected(e);
            }

        });

        viewer = new CheckboxTreeViewer(composite, SWT.BORDER | SWT.FULL_SELECTION );

        GridDataFactory.fillDefaults().grab(true, true).span(4, 1).applyTo(viewer.getControl());
        viewer.getControl().setToolTipText(Messages.DiagramLayersPage_SelectTT);
        viewer.setAutoExpandLevel(TreeViewer.ALL_LEVELS);
        viewer.getTree().setHeaderVisible(true);
        editor = new TreeEditor(viewer.getTree());

        final TreeColumn column1 = new TreeColumn(viewer.getTree(), SWT.LEFT);
        column1.setText(Messages.DiagramLayersPage_Role);
        column1.setWidth(100);
        final TreeColumn column2 = new TreeColumn(viewer.getTree(), SWT.LEFT);
        column2.setText(Messages.DiagramLayersPage_Show);
        column2.setWidth(50);
        final TreeColumn column3 = new TreeColumn(viewer.getTree(), SWT.LEFT);
        column3.setText(Messages.DiagramLayersPage_Focus);
        column3.setWidth(50);
        viewer.getTree().addListener(SWT.Resize, new Listener() {
            @Override
            public void handleEvent(Event event) {
                Tree tree = viewer.getTree();
                Point size = tree.getSize();
                int w = Math.max(size.x - 100 - tree.getBorderWidth() * 2, 30);
                column1.setWidth(w);
            }
        });

        viewer.getTree().addListener(SWT.PaintItem, new Listener() {
            public void handleEvent(Event event) {
                if ((event.index == 1 || event.index == 2) && !elements.isEmpty()) {
                    ILayer[] lz = layers.getLayers().toArray(new ILayer[0]);

                    TreeItem item = (TreeItem)event.item;
                    int index = viewer.getTree().indexOf(item);

                    int width = 0;
                    if (event.index == 1)
                        width = (column2.getWidth() - 1);
                    if (event.index == 2)
                        width = (column3.getWidth() - 1);

                    Attribute attribute = Attribute.Visible;
                    if (event.index == 2)
                        attribute = Attribute.Focusable;
                    Tristate state = getJointAttribute(elements, lz[index], attribute);

                    Color color = null;
                    if (state == null) {
                        color = viewer.getTree().getDisplay().getSystemColor(SWT.COLOR_GRAY);
                    } else {
                        switch (state) {
                            case False:
                                color = viewer.getTree().getDisplay().getSystemColor(SWT.COLOR_RED);
                                break;
                            case True:
                                color = viewer.getTree().getDisplay().getSystemColor(SWT.COLOR_GREEN);
                                break;
                            case Both:
                                color = viewer.getTree().getDisplay().getSystemColor(SWT.COLOR_GRAY);
                                break;
                        }
                    }

                    GC gc = event.gc;
                    Color foreground = gc.getForeground();
                    Color background = gc.getBackground();
                    gc.setBackground(color);
                    gc.setForeground(viewer.getTree().getDisplay().getSystemColor(SWT.COLOR_BLACK));
                    gc.fillRectangle(event.x, event.y, width-1, event.height-1);
                    Rectangle rect2 = new Rectangle(event.x, event.y, width-1, event.height-1);
                    gc.drawRectangle(rect2);
                    gc.setForeground(background);
                    gc.setBackground(foreground);
                }
            }
        });
        viewer.getTree().addKeyListener(new KeyAdapter() {
            @Override
            public void keyPressed(KeyEvent event) {
                if (event.keyCode == SWT.F2) {
                    // FIXME: Eclipse currently eats F2 presses. This should be
                    // implemented as a command handler or find some way to
                    // force these listeners to have priority...
                    TreeItem[] items = viewer.getTree().getSelection();
                    if(items.length != 1)
                        return;

                    TreeItem item = items[0];

                    ILayer layer = ISelectionUtils.filterSingleSelection(viewer.getSelection(), ILayer.class);
                    if (layer == null)
                        return;

                    startEditing(layer, item);
                } else if (event.keyCode == SWT.DEL) {
                    removeSelectedLayers();
                }
            }
        });
        viewer.getTree().addListener(SWT.MouseDown, new Listener() {
            public void handleEvent(Event event) {
                if (viewer.getControl().isDisposed())
                    return;

                Point pt = new Point(event.x, event.y);
                TreeItem item = viewer.getTree().getItem(pt);
                if (item == null)
                    return;

                int index = viewer.getTree().indexOf(item);
                ILayer[] lz = layers.getLayers().toArray(new ILayer[0]);
                ILayer layer = lz[index];

                Rectangle rect = item.getBounds(0);
                if (event.count == 2 && rect.contains(pt)) {
                    startEditing(layer, item);
                    return;
                }

                // Cannot adjust visibility/focusability if no elements are selected.
                if (elements.isEmpty())
                    return;

                rect = item.getBounds(1);
                if (rect.contains(pt)) {
                    Tristate state = getJointAttribute(elements, layer, Attribute.Visible);
                    setAttribute(elements, layer, Attribute.Visible, state.toggle().toBoolean());
                    refresh();
                    return;
                }

                Rectangle rect2 = item.getBounds(2);
                if (rect2.contains(pt)) {
                    Tristate state = getJointAttribute(elements, layer, Attribute.Focusable);
                    setAttribute(elements, layer, Attribute.Focusable, state.toggle().toBoolean());
                    refresh();
                    return;
                }
            }

            private void refresh() {
                viewer.getControl().redraw();
                context.getThreadAccess().asyncExec(new Runnable() {
                    @Override
                    public void run() {
                        if (context.isDisposed())
                            return;
                        context.getContentContext().setDirty();
                    }
                });
            }
        });

        viewer.addCheckStateListener(new ICheckStateListener(){
            @Override
            public void checkStateChanged(CheckStateChangedEvent event) {
                ILayer layer = (ILayer)event.getElement();
                if(event.getChecked()) layers.activate(layer);
                else layers.deactivate(layer);
                viewer.setSubtreeChecked(event.getElement(), event.getChecked());
                context.getThreadAccess().asyncExec(new Runnable() {

                    @Override
                    public void run() {
                        if(context.isDisposed()) return;
                        context.getContentContext().setDirty();
                    }

                });
            }
        });

        viewer.setContentProvider(new ITreeContentProvider(){
            @Override
            public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
            }
            @Override
            public void dispose() {
            }
            @Override
            public Object[] getElements(Object inputElement) {
                return layers.getLayers().toArray();
            }
            @Override
            public boolean hasChildren(Object element) {
                return false;
            }
            @Override
            public Object getParent(Object element) {
                return null;
            }
            @Override
            public Object[] getChildren(Object parentElement) {
                return new Object[0];
            }
        });
        viewer.setLabelProvider(new CellLabelProvider() {
            @Override
            public void update(ViewerCell cell) {
                if(cell.getColumnIndex() == 0) {
                    ILayer layer  = (ILayer)cell.getElement();
                    cell.setText(layer.getName());
                } else {
                    cell.setText(""); //$NON-NLS-1$
                }
            }
        });
        viewer.setCheckStateProvider(new ICheckStateProvider() {
            @Override
            public boolean isChecked(Object element) {
                ILayer layer  = (ILayer)element;
                final boolean isActive = layers.isActive(layer);
                return isActive;
            }
            @Override
            public boolean isGrayed(Object element) {
                return false;
            }
        });

        viewer.setInput(this);

        for(ILayer layer : layers.getVisibleLayers()) {
            viewer.setSubtreeChecked(layer, true);
        }

        composite.layout();
    }

    @Override
    public Control getControl() {
        return composite;
    }

    @Override
    public void setFocus() {
    }

    @Override
    public void addSelectionChangedListener(ISelectionChangedListener listener) {
    }

    @Override
    public ISelection getSelection() {
        return null;
    }

    @Override
    public void removeSelectionChangedListener(ISelectionChangedListener listener) {
    }

    @Override
    public void setSelection(ISelection selection) {
    }

    private void removeSelectedLayers() {
        TreeItem[] items = viewer.getTree().getSelection();
        if (items.length == 0)
            return;

        TreeItem[] all = viewer.getTree().getItems();
        int firstIndex = Arrays.indexOf(all, items[0]);
        for (TreeItem item : items) {
            int index = Arrays.indexOf(all, item);
            all[index] = null;
            ILayer layer = (ILayer)item.getData();
            layers.removeLayer(layer);
        }
        int selectIndex = firstIndex - 1;
        if (firstIndex == 0) {
            for (int i = firstIndex; i < all.length; ++i)
                if (all[i] != null) {
                    selectIndex = i;
                    break;
                }
        }
        if (selectIndex >= 0) {
            viewer.getTree().select(all[selectIndex]);
        }
        context.getThreadAccess().asyncExec(new Runnable() {

            @Override
            public void run() {
                if(context.isDisposed()) return;
                context.getContentContext().setDirty();
            }

        });
    }

    private boolean startEditing(final ILayer layer, final TreeItem item/*, final int columnIndex*/) {

        //        Column column = columns[columnIndex];

        String initialText = layer.getName();

        final Composite composite = new Composite(viewer.getTree(), SWT.NONE);
        final Text text = new Text(composite, SWT.BORDER);
        final int insetX = 0;
        final int insetY = 0;
        composite.addListener(SWT.Resize, new Listener() {
            public void handleEvent(Event e) {
                Rectangle rect = composite.getClientArea();
                text.setBounds(rect.x + insetX, rect.y + insetY, rect.width - insetX * 2, rect.height
                        - insetY * 2);
            }
        });
        Listener textListener = new Listener() {
            public void handleEvent(final Event e) {
                //String error;
                String newText;
                switch (e.type) {
                    case SWT.FocusOut:
                        if(layer instanceof IEditableLayer) {
                            IEditableLayer l = (IEditableLayer)layer;
                            String name = findFreshName(layers, text.getText());
                            l.setName(name);
                            viewer.refresh();
                        }

                        //        				// Item may be disposed if the tree gets reset after a previous editing.
                        //        				if (!item.isDisposed()) {
                        //        					item.setText(columnIndex, text.getText());
                        //        					queueSelectionRefresh(context);
                        //        				}
                        //        			} else {
                        //        				//                                System.out.println("validation error: " + error);
                        //        			}
                        composite.dispose();
                        break;
                    case SWT.Modify:
                        //        			newText = text.getText();
                        //        			error = modifier.isValid(newText);
                        //        			if (error != null) {
                        //        				text.setBackground(invalidModificationColor);
                        //        				//                                System.out.println("validation error: " + error);
                        //        			} else {
                        //        				text.setBackground(null);
                        //        			}
                        break;
                    case SWT.Verify:
                        newText = text.getText();
                        String leftText = newText.substring(0, e.start);
                        String rightText = newText.substring(e.end, newText.length());
                        GC gc = new GC(text);
                        Point size = gc.textExtent(leftText + e.text + rightText);
                        gc.dispose();
                        size = text.computeSize(size.x, SWT.DEFAULT);
                        editor.horizontalAlignment = SWT.LEFT;
                        Rectangle itemRect = item.getBounds(0),
                        rect = viewer.getTree().getClientArea();
                        editor.minimumWidth = Math.max(size.x, itemRect.width) + insetX * 2;
                        int left = itemRect.x,
                        right = rect.x + rect.width;
                        editor.minimumWidth = Math.min(editor.minimumWidth, right - left);
                        editor.minimumHeight = size.y + insetY * 2;
                        editor.layout();
                        break;
                    case SWT.Traverse:
                        switch (e.detail) {
                            case SWT.TRAVERSE_RETURN:
                                if(layer instanceof IEditableLayer) {
                                    IEditableLayer l = (IEditableLayer)layer;
                                    String name = findFreshName(layers, text.getText());
                                    l.setName(name);
                                    viewer.refresh();
                                    //System.out.println("renamed layer to " + text.getText());
                                    //viewer.refresh();
                                }
                                //        				error = modifier.isValid(text.getText());
                                //        				if (error == null) {
                                //        					modifier.modify(text.getText());
                                //        					if (!item.isDisposed()) {
                                //        						item.setText(columnIndex, text.getText());
                                //        						queueSelectionRefresh(context);
                                //        					}
                                //        				}
                                // FALL THROUGH
                            case SWT.TRAVERSE_ESCAPE:
                                composite.dispose();
                                e.doit = false;
                                break;
                            default:
                                //System.out.println("unhandled traversal: " + e.detail);
                                break;
                        }
                        break;
                }
            }
        };
        text.addListener(SWT.FocusOut, textListener);
        text.addListener(SWT.Traverse, textListener);
        text.addListener(SWT.Verify, textListener);
        text.addListener(SWT.Modify, textListener);
        editor.setEditor(composite, item, 0);
        text.setText(initialText);
        text.selectAll();
        text.setFocus();
        //        lastItem[0] = item;
        return true;
    }

}
