/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.diagram;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.IndexRoot;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.synchronization.graph.DiagramGraphUtil;
import org.simantics.modeling.ui.preferences.DiagramPreferences;
import org.simantics.scl.commands.Commands;
import org.simantics.utils.page.PageDesc;

/**
 * A dialog for changing page settings
 * 
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 */
public class PageSettingsDialog extends Dialog {

    private final Session session;
    private final Resource diagramResource;

    private PageDescComposite pdc;
    private Text gridSizeText;
    private Button showBordersButton;
    private Button showMarginsButton;

    private PageDesc pageDesc;
    private double gridSize;
    private double lastGridSize;
    private boolean borders = false;
    private boolean margins = false;


    public PageSettingsDialog(Resource diagramResource, Shell parentShell) {
        super(parentShell);
        this.session = Simantics.getSession();
        this.diagramResource = diagramResource;
    }


    @Override
    protected void configureShell(Shell newShell) {
        super.configureShell(newShell);
        newShell.setText(Messages.PageSettingsDialog_PageSettings);
    }

    @Override
    protected Control createDialogArea(Composite parent) {
        Composite composite = (Composite) super.createDialogArea(parent);

        Group group = new Group(composite, SWT.NONE);
        group.setText(Messages.PageSettingsDialog_GridSnapping);
        GridDataFactory.fillDefaults().grab(true, false).span(1, 1).applyTo(group);
        Label label = new Label(group, SWT.NONE);
        label.setText(Messages.PageSettingsDialog_GridSizeMM);
        gridSizeText = new Text(group, SWT.SINGLE|SWT.BORDER);
        GridLayoutFactory.fillDefaults().numColumns(2).equalWidth(false).extendedMargins(12, 12, 12, 12).spacing(5, 4).applyTo(group);
        GridDataFactory.fillDefaults().grab(true, false).span(1, 1).applyTo(gridSizeText);

        Group pageGroup = new Group(composite, SWT.NONE);
        pageGroup.setText(Messages.PageSettingsDialog_PageSize);
        pdc = new PageDescComposite(pageGroup, SWT.NONE);
        GridDataFactory.fillDefaults().grab(true, false).span(1, 1).applyTo(pageGroup);
        GridLayoutFactory.fillDefaults().numColumns(1).equalWidth(false).extendedMargins(12, 12, 12, 12).spacing(5, 4).applyTo(pageGroup);
        GridDataFactory.fillDefaults().grab(true, true).span(1, 1).applyTo(pdc);

        Group displayGroup = new Group(composite, SWT.NONE);
        displayGroup.setText(Messages.PageSettingsDialog_Display);
        showBordersButton = new Button(displayGroup, SWT.CHECK);
        showBordersButton.setText(Messages.PageSettingsDialog_ShowPageBorders);
        showMarginsButton = new Button(displayGroup, SWT.CHECK);
        showMarginsButton.setText(Messages.PageSettingsDialog_ShowMargins);
        GridLayoutFactory.fillDefaults().numColumns(1).equalWidth(false).extendedMargins(12, 12, 12, 12).spacing(5, 4).applyTo(displayGroup);

        GridDataFactory.fillDefaults().grab(true, false).span(1, 1).applyTo(displayGroup);

        gridSizeText.addModifyListener(new ModifyListener() {
            @Override
            public void modifyText(ModifyEvent e) {
                String text = ((Text)e.widget).getText();
                try {
                    gridSize = Double.parseDouble(text);
                    if (gridSize < 1e-3 || gridSize > 1e3) {
                        gridSize = lastGridSize;
                        setGridSizeValid(false);
                    } else {
                        setGridSizeValid(true);
                    }
                } catch (NumberFormatException err) {
                    gridSize = lastGridSize;
                    setGridSizeValid(false);
                }
            }

            void setGridSizeValid(boolean valid) {
                if (valid) {
                    gridSizeText.setBackground(null);
                    gridSizeText.setForeground(null);
                } else {
                    //gridSizeText.setBackground(gridSizeText.getDisplay().getSystemColor(SWT.COLOR_RED));
                    gridSizeText.setForeground(gridSizeText.getDisplay().getSystemColor(SWT.COLOR_RED));
                }
            }
        });

        showBordersButton.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                borders = ((Button)e.widget).getSelection();
            }
        });

        showMarginsButton.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                margins = ((Button)e.widget).getSelection();
            }
        });

        loadValues();

        return composite;
    }

    public void applySettings() {
        if (pdc.getPageDesc() != null)
            pageDesc = pdc.getPageDesc();
        session.markUndoPoint();
        session.asyncRequest(new WriteRequest() {
            @Override
            public void perform(WriteGraph graph) throws DatabaseException {
                Resource model = graph.syncRequest(new IndexRoot(diagramResource));
                Commands.get(graph, "Simantics/PageSettings/setPageDesc") //$NON-NLS-1$
                        .execute(graph, model, diagramResource, pageDesc.toRepr());
                Commands.get(graph, "Simantics/PageSettings/setGridSize") //$NON-NLS-1$
                        .execute(graph, model, diagramResource, gridSize);
                Commands.get(graph, "Simantics/PageSettings/setPageBordersVisible") //$NON-NLS-1$
                        .execute(graph, model, diagramResource, borders);
                Commands.get(graph, "Simantics/PageSettings/setMarginsVisible") //$NON-NLS-1$
                        .execute(graph, model, diagramResource, margins);
            }
        });
    }

    private void loadValues() {
        session.asyncRequest(new ReadRequest() {

            @Override
            public void run(ReadGraph graph) throws DatabaseException {
                pageDesc = DiagramGraphUtil.getPageDesc(graph, diagramResource, PageDesc.DEFAULT);
                gridSize = DiagramGraphUtil.getGridSize(graph, diagramResource, DiagramPreferences.DEFAULT_SNAP_GRID_SIZE);
                lastGridSize = gridSize;
                borders = DiagramGraphUtil.isPageBordersVisible(graph, diagramResource);
                margins = DiagramGraphUtil.isMarginsVisible(graph, diagramResource);
                Display.getDefault().asyncExec(new Runnable() {
                    @Override
                    public void run() {
                        applyValues();
                    }
                });
            }
        });
    }

    private void applyValues() {
        pdc.setPageDesc(pageDesc);
        gridSizeText.setText(Double.toString(gridSize));
        showBordersButton.setSelection(borders);
        showMarginsButton.setSelection(margins);
    }

}
