package org.simantics.modeling.ui.componentTypeEditor;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Arrays;
import java.util.Collections;
import java.util.Set;
import java.util.stream.Collectors;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.operation.IRunnableContext;
import org.eclipse.jface.text.Document;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.Position;
import org.eclipse.jface.text.source.Annotation;
import org.eclipse.jface.text.source.AnnotationModel;
import org.eclipse.jface.text.source.IAnnotationModel;
import org.eclipse.ui.texteditor.AbstractDocumentProvider;
import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.ActiveModels;
import org.simantics.db.layer0.request.ActiveRuns;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.db.procedure.Listener;
import org.simantics.db.request.Read;
import org.simantics.modeling.ComponentTypeScriptRequest;
import org.simantics.modeling.ComponentTypeScriptResult;
import org.simantics.scl.compiler.errors.CompilationError;
import org.simantics.scl.compiler.errors.Locations;
import org.simantics.scl.ui.editor2.SCLCodeHighlighting;
import org.simantics.simulation.experiment.IExperiment;
import org.simantics.simulation.project.IExperimentManager;
import org.simantics.structural.stubs.StructuralResource2;
import org.simantics.structural2.utils.StructuralUtils;
import org.simantics.ui.workbench.ResourceEditorInput;
import org.simantics.utils.logging.TimeLogger;
import org.simantics.utils.strings.EString;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ComponentTypeScriptDocumentProvider extends AbstractDocumentProvider {

    private static final Logger LOGGER = LoggerFactory.getLogger(ComponentTypeScriptDocumentProvider.class);

    protected ComponentTypeScriptEditor editor;
    
    protected Resource resource;
    protected String currentText;
    protected boolean immutable;
    protected boolean errorHappened;
    
    protected AnnotationModel annotationModel = new AnnotationModel();
    
    protected boolean annotationsInitialized = false;
    
    public ComponentTypeScriptDocumentProvider(ComponentTypeScriptEditor editor) {
        this.editor = editor;
    }

    @Override
    protected IDocument createDocument(Object element) throws CoreException {
        ResourceEditorInput input = (ResourceEditorInput)element;
        resource = input.getResource();
        try {
            var doc = Simantics.getSession().syncRequest(new UniqueRead<Document>() {
                @Override
                public Document perform(ReadGraph graph) throws DatabaseException {
                    StructuralResource2 STR = StructuralResource2.getInstance(graph);
                    currentText = graph.getRelatedValue(resource, STR.ComponentTypeScript_code, Bindings.STRING);
                    Resource owner = graph.getPossibleObject(resource, STR.ComponentType_hasScript_Inverse);
                    immutable = Layer0Utils.isMarkedReadOnly(graph, resource)
                            || owner != null && StructuralUtils.isImmutable(graph, owner);
                    errorHappened = false;
                    return new Document(currentText != null ? currentText : ""); //$NON-NLS-1$
                }
            });
            SCLCodeHighlighting.connectPartitioner(doc);
            return doc;
        } catch (DatabaseException e) {
            StringWriter sw = new StringWriter();
            PrintWriter pw = new PrintWriter(sw);
            e.printStackTrace(pw);
            errorHappened = true;
            return new Document(sw.toString());
        }
    }

    private static class ActiveExperimentIdentifiersRequest extends UniqueRead<Set<String>> {

        private static final ActiveExperimentIdentifiersRequest INSTANCE = new ActiveExperimentIdentifiersRequest();

        @Override
        public Set<String> perform(ReadGraph graph) throws DatabaseException {
            // This relies on ExperimentManager returning IExperiments only when they are completely activated.
            IExperimentManager em = Simantics.getProject().getHint(IExperimentManager.KEY_EXPERIMENT_MANAGER);
            var activeExperimentIds = Arrays.stream(em.getExperiments())
                    .map(IExperiment::getIdentifier)
                    .collect(Collectors.toSet());

            // This walks through experiments marked as active in the database.
            // It is needed purely to create a dependency on the modeled active
            // experiment structures in the database to track database changes.
            for (var root : graph.syncRequest(new ActiveModels(Simantics.getProjectResource()))) {
                graph.syncRequest(new ActiveRuns(root));
            }

            return activeExperimentIds.isEmpty() ? Collections.emptySet() : activeExperimentIds;
        }

    }

    protected void updateAnnotations() {
        Simantics.getSession().asyncRequest(new Read<ComponentTypeScriptResult>() {
            @Override
            public ComponentTypeScriptResult perform(ReadGraph graph) throws DatabaseException {
                // is this the correct way to obtain the parent?
                StructuralResource2 str = StructuralResource2.getInstance(graph);
                Resource componentType = graph.getSingleObject(resource, str.ComponentType_hasScript_Inverse);
                // #889: Include the set of active experiment IDs in ComponentTypeScriptRequest
                // to ensure the request is really re-performed when the active experiment changes
                // since the active experiment can affect user component SCL script validation.
                Object activeExperiments = graph.syncRequest(ActiveExperimentIdentifiersRequest.INSTANCE);
                ComponentTypeScriptResult result = graph.syncRequest(new ComponentTypeScriptRequest(resource, componentType, activeExperiments));
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("updateAnnotations request.perform:\n\tactive experiments: {}\r\terrors: {}", activeExperiments, result.getErrors());
                }
                return result;
            }
        }, new Listener<ComponentTypeScriptResult>() {

            @Override
            public void execute(ComponentTypeScriptResult result) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("updateAnnotations listener.execute: errors:\n\t{}", EString.implode(result.getErrors(), "\n\t"));
                }
                synchronized(annotationModel.getLockObject()) {
                    annotationModel.removeAllAnnotations();
                    for(CompilationError error : result.getErrors()) {
                        Annotation annotation = new Annotation("org.eclipse.ui.workbench.texteditor.error", true, error.description); //$NON-NLS-1$
                        int begin = Locations.beginOf(error.location);
                        int end = Locations.endOf(error.location);
                        Position position = new Position(begin, end - begin);
                        annotationModel.addAnnotation(annotation, position);
                    }
                }
            }

            @Override
            public void exception(Throwable t) {
                LOGGER.error("Failed to update component type script annotations for input resource {}", resource, t);
            }

            @Override
            public boolean isDisposed() {
                return editor.isDisposed();
            }
        });
    }
    
    @Override
    protected void doSaveDocument(IProgressMonitor monitor, Object element,
            IDocument document, boolean overwrite) throws CoreException {
        TimeLogger.resetTimeAndLog(getClass(), "doSaveDocument"); //$NON-NLS-1$
        currentText = document.get();
        Simantics.getSession().asyncRequest(new WriteRequest() {
            @Override
            public void perform(WriteGraph graph) throws DatabaseException {
                graph.markUndoPoint();
                StructuralResource2 STR = StructuralResource2.getInstance(graph);
                graph.claimLiteral(resource, STR.ComponentTypeScript_code, currentText, Bindings.STRING);
            }
        });
    }

    @Override
    protected IAnnotationModel createAnnotationModel(Object element) throws CoreException {
        if(!annotationsInitialized) {
            updateAnnotations();
            annotationsInitialized = true;
        }
        return annotationModel;
    }

    @Override
    protected IRunnableContext getOperationRunner(IProgressMonitor monitor) {
        return null;
    }
    
    @Override
    public boolean isModifiable(Object element) {
        return !errorHappened && !immutable;
    }

    @Override
    public boolean isReadOnly(Object element) {
        return errorHappened || immutable;
    }

}
