/*******************************************************************************
 * Copyright (c) 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.property.svg;

import java.util.Collection;

import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.resource.ColorDescriptor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchSite;
import org.simantics.browsing.ui.swt.widgets.DefaultColorProvider;
import org.simantics.browsing.ui.swt.widgets.Label;
import org.simantics.browsing.ui.swt.widgets.TrackedText;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupport;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.management.ISessionContext;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.image2.ontology.ImageResource;
import org.simantics.selectionview.ComparableTabContributor;
import org.simantics.selectionview.ConfigurationComposite;
import org.simantics.selectionview.TabContribution;
import org.simantics.ui.selection.WorkbenchSelectionUtils;

/**
 * @author Tuukka Lehtonen
 */
public class SVGElementComposite extends ConfigurationComposite {

	public static TabContribution<Object> svgTabContribution() {
		return new TabContribution<Object>() {

			@Override
			public boolean accept(ReadGraph graph, Object input) throws DatabaseException {
				Resource res = WorkbenchSelectionUtils.getPossibleResourceFromSelection(graph, input);
				if(res == null)
					return false;
				if(graph.isInstanceOf(res, DiagramResource.getInstance(graph).SVGElement))
					return true;
				if(graph.isInstanceOf(res, ImageResource.getInstance(graph).SvgImage))
					return true;
				return false;
			}

			@Override
			public void contribute(ReadGraph graph, Object selection, Collection<ComparableTabContributor> result)
					throws DatabaseException {
				Resource res = WorkbenchSelectionUtils.getPossibleResourceFromSelection(graph, selection);
				if(res != null) {
					if(graph.isInstanceOf(res, DiagramResource.getInstance(graph).SVGElement)) {
						result.add(make(res, 100.0, "SVG"));
					}
					if(graph.isInstanceOf(res, ImageResource.getInstance(graph).SvgImage)) {
						result.add(make(res, 100.0, "SVG"));
					}
				}
			}
		};
	}
	
    public static ComparableTabContributor make(Resource input, double priority, String tabLabel) {
        return make(new SVGInput(input), priority, tabLabel);
    }

    public static ComparableTabContributor make(SVGInput input, double priority, String tabLabel) {
        return new ComparableTabContributor(new SVGElementComposite(), priority, input, tabLabel);
    }

    public void create(final Composite body, IWorkbenchSite site, final ISessionContext context, final WidgetSupport support) {

        final Display display = body.getDisplay();

        body.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

        Label expressionHeader = new Label(body, support, 0);
        expressionHeader.setText("SVG markup");
        expressionHeader.setFont(smallFont);
        expressionHeader.setBackground(display.getSystemColor(SWT.COLOR_WHITE));
        GridDataFactory.fillDefaults().grab(true, false).span(2, 1).align(SWT.CENTER, SWT.CENTER).applyTo(expressionHeader.getWidget());

        TrackedText expression = new TrackedText(body, support, SWT.MULTI | SWT.BORDER | SWT.V_SCROLL);
        expression.setSelectAllOnStartEdit(false);
        expression.setColorProvider(new DefaultColorProvider(expression.getResourceManager()) {

            private final ColorDescriptor inactiveColor = ColorDescriptor.createFrom(new RGB(220, 220, 255));

            @Override
            public Color getInactiveBackground() {
                return resourceManager.createColor(inactiveColor);
            }

        });
        expression.setTextFactory(new SVGTextFactory());
        expression.addModifyListener(new SVGModifier());
        expression.setInputValidator(new IInputValidator() {
            @Override
            public String isValid(String document) {
                try {
                    XMLPrettyPrinter.parseDocument(document);
                    return null;
                } catch (Exception e) {
                    return e.getMessage();
                }
            }
        });
        GridDataFactory.fillDefaults().grab(true, true).applyTo(expression.getWidget());

    }

}
