/*******************************************************************************
 * Copyright (c) 2007, 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.preferences;

import java.nio.charset.Charset;

import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.preference.BooleanFieldEditor;
import org.eclipse.jface.preference.ComboFieldEditor;
import org.eclipse.jface.preference.FieldEditor;
import org.eclipse.jface.preference.FieldEditorPreferencePage;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.IntegerFieldEditor;
import org.eclipse.jface.preference.StringFieldEditor;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;
import org.eclipse.ui.preferences.ScopedPreferenceStore;
import org.simantics.history.csv.ColumnSeparator;
import org.simantics.history.csv.DecimalSeparator;
import org.simantics.history.csv.ExportInterpolation;
import org.simantics.modeling.preferences.CSVPreferences;

public class CSVPreferencePage extends FieldEditorPreferencePage implements IWorkbenchPreferencePage {

    private ComboFieldEditor fDecimalSeparatorEditor = null;
    private ComboFieldEditor fColumnSeparatorEditor = null;
    private StringFieldEditor fExtensionEditor = null;
    private BooleanFieldEditor fResamplingEditor = null;
    private ComboFieldEditor fSamplingModeEditor = null;
    private StringFieldEditor fStartEditor = null;
    private StringFieldEditor fStepEditor = null;
    private IntegerFieldEditor fTimeDigitsEditor = null;
    private IntegerFieldEditor fFloatDigitsEditor = null;
    private IntegerFieldEditor fDoubleDigitsEditor = null;
    private StringFieldEditor fOutputEncoding = null;

    private String decimalSeparatorValue;
    private String columnSeparatorValue;

    public CSVPreferencePage() {
        super(GRID);
        setDescription("CSV import and export preferences");
        IPreferenceStore pf = new ScopedPreferenceStore(InstanceScope.INSTANCE, CSVPreferences.P_NODE);
        setPreferenceStore( pf );	
    }

    @Override
    public void createControl(Composite parent) {
        super.createControl(parent);
    }

    @Override
	public void init(IWorkbench workbench) {
	}

    @Override
    protected void initialize() {
        super.initialize();

        if (decimalSeparatorValue.equals(columnSeparatorValue)) {
            setErrorMessage("Column and decimal separators must not be the same.");
            checkState();
            updateApplyButton();
        }
    }

	@Override
	protected void createFieldEditors() {
	    decimalSeparatorValue = getPreferenceStore().getString(CSVPreferences.P_CSV_DECIMAL_SEPARATOR);
	    columnSeparatorValue = getPreferenceStore().getString(CSVPreferences.P_CSV_COLUMN_SEPARATOR);

	    String[][] decimalSeparators = new String[DecimalSeparator.values().length][];
	    int i = 0;
	    for (DecimalSeparator ds : DecimalSeparator.values())
	        decimalSeparators[i++] = new String[] { ds.label, ds.preference };
	    String[][] columnSeparators = new String[ColumnSeparator.values().length][];
	    i = 0;
	    for (ColumnSeparator cs : ColumnSeparator.values())
	        columnSeparators[i++] = new String[] { cs.label, cs.preference };

	    fDecimalSeparatorEditor = new ComboFieldEditor(CSVPreferences.P_CSV_DECIMAL_SEPARATOR, "Decimal separator", decimalSeparators, getFieldEditorParent()) {
	        @Override
	        public boolean isValid() {
	            return !decimalSeparatorValue.equals(columnSeparatorValue);
	        }
	    };
	    addField(fDecimalSeparatorEditor);

	    fColumnSeparatorEditor = new ComboFieldEditor(CSVPreferences.P_CSV_COLUMN_SEPARATOR, "Column separator", columnSeparators, getFieldEditorParent()) {
            @Override
            public boolean isValid() {
                return !decimalSeparatorValue.equals(columnSeparatorValue);
            }
        };

	    addField(fColumnSeparatorEditor);

	    fExtensionEditor = new StringFieldEditor(CSVPreferences.P_CSV_FILE_EXTENSION, "File extension", getFieldEditorParent());
	    addField(fExtensionEditor);
	    
	    fResamplingEditor = new BooleanFieldEditor(CSVPreferences.P_CSV_RESAMPLE, "&Resample", getFieldEditorParent());
	    fResamplingEditor.getDescriptionControl(getFieldEditorParent()).setToolTipText("Resample exported data from raw data");
	    addField(fResamplingEditor);

	    fSamplingModeEditor = new ComboFieldEditor(CSVPreferences.P_CSV_SAMPLING_MODE, "Sampling mode",
	            new String[][] {
	    		{ExportInterpolation.LINEAR_INTERPOLATION.label, ExportInterpolation.LINEAR_INTERPOLATION.preference},
	    		{ExportInterpolation.PREVIOUS_SAMPLE.label, ExportInterpolation.PREVIOUS_SAMPLE.preference},
	    		}, getFieldEditorParent());
	    addField(fSamplingModeEditor);

	    fStartEditor = new StringFieldEditor(CSVPreferences.P_CSV_START_TIME, "Start time (s)", getFieldEditorParent()) {
	    	protected boolean doCheckState() {
	            String text = getTextControl().getText();
	            if (text == null || text.isEmpty()) {
	                return true;
	            }
	            try {
	                @SuppressWarnings("unused")
	                double number = Double.parseDouble(text);
	                return true;
	            } catch (NumberFormatException e1) {
	                return false;
	            }
	    	};
	    };
	    fStartEditor.setEmptyStringAllowed(true);
	    fStartEditor.setErrorMessage("Enter valid start time value (s)");
	    fStartEditor.getTextControl(getFieldEditorParent()).setToolTipText("Time to start resampling from");
	    addField(fStartEditor);

	    fStepEditor = new StringFieldEditor(CSVPreferences.P_CSV_TIME_STEP, "Step size (s)", getFieldEditorParent()) {
	    	protected boolean doCheckState() {
	            String text = getTextControl().getText();
	            if (text == null || text.isEmpty()) {
	                return true;
	            }
	            try {
	                double number = Double.parseDouble(text);
	                return number>0;
	            } catch (NumberFormatException e1) {
	                return false;
	            }
	    	};
	    };
	    fStepEditor.setEmptyStringAllowed(true);
	    fStepEditor.setErrorMessage("Enter valid time step value (s)");
	    fStepEditor.getTextControl(getFieldEditorParent()).setToolTipText("Time step to use when resampling");
	    addField(fStepEditor);

	    Group significantDigitsGroup = new Group(getFieldEditorParent(), SWT.NONE);
	    significantDigitsGroup.setText("Significant digits");
	    significantDigitsGroup.setToolTipText("Control the Amount of Significant Digits in Exported Numbers");
	    GridDataFactory.fillDefaults().indent(0, 5).span(2, 1).applyTo(significantDigitsGroup);

	    fTimeDigitsEditor = new IntegerFieldEditor(CSVPreferences.P_CSV_TIME_DIGITS, "Time stamps", significantDigitsGroup, 2);
	    fTimeDigitsEditor.setValidRange(1, 15);
	    addField(fTimeDigitsEditor);
	    fFloatDigitsEditor = new IntegerFieldEditor(CSVPreferences.P_CSV_FLOAT_DIGITS, "Single precision floating point", significantDigitsGroup, 1);
	    fFloatDigitsEditor.setValidRange(1, 7);
	    addField(fFloatDigitsEditor);
	    fDoubleDigitsEditor = new IntegerFieldEditor(CSVPreferences.P_CSV_DOUBLE_DIGITS, "Double precision floating point", significantDigitsGroup, 2);
	    fDoubleDigitsEditor.setValidRange(1, 15);
	    addField(fDoubleDigitsEditor);

	    GridLayoutFactory.swtDefaults().numColumns(2).applyTo(significantDigitsGroup);

	    fOutputEncoding = new StringFieldEditor(CSVPreferences.P_CSV_OUTPUT_ENCODING, "Output encoding", getFieldEditorParent()) {
	        @Override
	        protected boolean doCheckState() {
	            try {
	                Charset.forName(getStringValue());
	                return true;
	            } catch (IllegalArgumentException e) {
	                return false;
	            }
	        }
	    };
	    fOutputEncoding.setEmptyStringAllowed(false);
	    fOutputEncoding.setValidateStrategy(StringFieldEditor.VALIDATE_ON_KEY_STROKE);
	    fOutputEncoding.setErrorMessage("Enter valid encoding / character set name");
	    addField(fOutputEncoding);

	    updateSampling(getPreferenceStore().getBoolean(CSVPreferences.P_CSV_RESAMPLE));
	}

    /**
     * @see org.eclipse.jface.util.IPropertyChangeListener#propertyChange(org.eclipse.jface.util.PropertyChangeEvent)
     */
    @Override
    public void propertyChange(PropertyChangeEvent event) {
        if (event.getProperty().equals(FieldEditor.VALUE)) {
            Object source = event.getSource();
            boolean validate = false;
            if (source == fDecimalSeparatorEditor) {
                decimalSeparatorValue = (String) event.getNewValue();
                validate = true;
            } else if (source == fColumnSeparatorEditor) {
                columnSeparatorValue = (String) event.getNewValue();
                validate = true;
            } else if (source == fResamplingEditor) {
                updateSampling();
            }
            if (validate) {
                if (decimalSeparatorValue.equals(columnSeparatorValue)) {
                    setErrorMessage("Column and decimal separators must not be the same.");
                } else {
                    setErrorMessage(null);
                }
                checkState();
                updateApplyButton();
            }
        }
        super.propertyChange(event);
    }

    private void updateSampling() {
        updateSampling(fResamplingEditor.getBooleanValue());
    }

    private void updateSampling(boolean resample) {
        Label fStartLabel = fStartEditor.getLabelControl(getFieldEditorParent());
        Label fStepLabel = fStepEditor.getLabelControl(getFieldEditorParent());
        Color gray = fStartLabel.getDisplay().getSystemColor(SWT.COLOR_DARK_GRAY);
        fStartLabel.setForeground(resample ? null : gray);
        fStepLabel.setForeground(resample ? null : gray);
    }

}
