/*******************************************************************************
 * Copyright (c) 2007, 2017 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - (#7084) refactoring
 *******************************************************************************/
package org.simantics.modeling.ui.pdf;

import java.util.concurrent.Semaphore;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.common.request.PossibleIndexRoot;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ValidationException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.db.request.Read;
import org.simantics.diagram.elements.DiagramNodeUtil;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.g2d.canvas.Hints;
import org.simantics.g2d.canvas.impl.CanvasContext;
import org.simantics.g2d.scenegraph.ICanvasSceneGraphProvider;
import org.simantics.modeling.requests.Node;
import org.simantics.structural.stubs.StructuralResource2;
import org.simantics.utils.datastructures.Pair;
import org.simantics.utils.page.PageDesc;
import org.simantics.utils.threads.IThreadWorkQueue;
import org.simantics.utils.threads.ThreadUtils;

import com.lowagie.text.Rectangle;
import com.lowagie.text.pdf.FontMapper;
import com.lowagie.text.pdf.PdfWriter;

/**
 * Rasterizes diagram into a PDF document using the com.lowagie.text plug-in.
 * 
 * @author Tuukka Lehtonen
 */
public class PDFPainter {

    public static void render(
            final IThreadWorkQueue thread,
            PDFExportPlan exportModel,
            final Node node,
            final PdfWriter writer,
            final FontMapper mapper,
            final Rectangle pageSize,
            final PageDesc pageDesc,
            final boolean fitDiagramContentsToPageMargins,
            long timeout)
                    throws InterruptedException, DatabaseException
    {
        DatabaseException[] exception = { null };
        ICanvasSceneGraphProvider[] sgProvider = { null };

        CanvasContext ctx = new CanvasContext(thread);
        ctx.getDefaultHintContext().setHint(Hints.KEY_DISABLE_GRAPH_MODIFICATIONS, Boolean.TRUE);

        try {
            final Semaphore done = new Semaphore(0);
            // IMPORTANT: Load diagram in a different thread than the canvas context thread!
            ThreadUtils.getBlockingWorkExecutor().execute(() -> {
                try {
                    Session s = exportModel.sessionContext.getSession();

                    Pair<Resource, String> modelAndRVI = s.syncRequest( modelAndRVI(node) );
                    Boolean isSymbol = s.syncRequest( isSymbol(node) ); 

                    ICanvasSceneGraphProvider provider = DiagramNodeUtil.loadSceneGraphProvider(
                            ctx,
                            modelAndRVI.first,
                            node.getDiagramResource(),
                            modelAndRVI.second,
                            5000);
                    sgProvider[0] = provider;
                    ctx.getDefaultHintContext().setHint(Hints.KEY_PAGE_DESC, pageDesc);

//                    System.err.println(NodeUtil.printTreeNodes(ctx.getCanvasNode(), new StringBuilder()).toString());

                    ThreadUtils.asyncExec(thread, () -> {
                        try {
                            boolean fitToContent = fitDiagramContentsToPageMargins || isSymbol;
                            if (!fitToContent) {
                                // Prevent PDF printing from drawing page borders if the
                                // print area is fitted directly to the page size.
                                // This avoids unwanted black half-visible edges.
                                ctx.getDefaultHintContext().setHint(Hints.KEY_DISPLAY_PAGE, false);
                            }

                            PDFBuilder chassis = new PDFBuilder(writer, mapper, pageSize, pageDesc, fitToContent);
                            chassis.paint(ctx, true);
                        } catch (Throwable e) {
                            exception[0] = new DatabaseException(e);
                        } finally {
                            done.release();
                        }
                    });
                } catch (DatabaseException e) {
                    done.release();
                    exception[0] = e;
                } catch (Throwable e) {
                    done.release();
                    exception[0] = new DatabaseException(e);
                } finally {
                    done.release();
                }
            });

            done.acquire(2);
            if (exception[0] != null)
                throw exception[0];
        } finally {
            if (sgProvider[0] != null)
                sgProvider[0].dispose();
            ctx.dispose();
        }
    }

    private static Read<Pair<Resource, String>> modelAndRVI(Node node) {
        return new UniqueRead<Pair<Resource, String>>() {
            @Override
            public Pair<Resource, String> perform(ReadGraph graph) throws DatabaseException {
                return Pair.make( resolveModel(graph, node), resolveRVI(graph, node) );
            }
        };
    }

    private static Read<Boolean> isSymbol(Node node) {
        return new UniqueRead<Boolean>() {
            @Override
            public Boolean perform(ReadGraph graph) throws DatabaseException {
                StructuralResource2 STR = StructuralResource2.getInstance(graph);
                DiagramResource DIA = DiagramResource.getInstance(graph);
                Resource possibleSymbol = graph.getPossibleObject(node.getDiagramResource(), STR.Defines);
                return possibleSymbol != null && graph.isInstanceOf(possibleSymbol, DIA.ElementClass);
            }
        };
    }

    private static Resource resolveModel(ReadGraph graph, Node node) throws DatabaseException {
        Resource composite = node.getDefiningResources().head();
        Resource model = graph.syncRequest(new PossibleIndexRoot(composite));
        if (model == null)
            throw new ValidationException("no model found for composite " + NameUtils.getSafeName(graph, composite));
        return model;
    }

    private static String resolveRVI(ReadGraph graph, final Node node) throws DatabaseException {
        String RVI = node.getRVI();
        if (RVI != null) return RVI;
        Resource composite = node.getDefiningResources().head();
        Variable var = Variables.getVariable(graph, composite);
        org.simantics.db.layer0.variable.RVI rvi = var.getPossibleRVI(graph);
        return rvi != null ? rvi.toString() : null;
    }

}