/*******************************************************************************
 * Copyright (c) 2017, 2023 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - #7297
 *******************************************************************************/
package org.simantics.modeling.ui.pdf;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiPredicate;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.layout.TreeColumnLayout;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.resource.LocalResourceManager;
import org.eclipse.jface.viewers.CellLabelProvider;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.ICheckStateProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.TreeItem;
import org.simantics.browsing.ui.common.views.DefaultFilterStrategy;
import org.simantics.browsing.ui.common.views.IFilterStrategy;
import org.simantics.modeling.requests.CollectionResult;
import org.simantics.modeling.requests.Node;
import org.simantics.modeling.requests.Nodes;
import org.simantics.utils.strings.AlphanumComparator;
import org.simantics.utils.ui.ISelectionUtils;

/**
 * A tree of nodes intended for usable listing and selecting diagrams.
 * 
 * @author Tuukka Lehtonen
 * @since 1.30.0
 */
public class NodeTree extends Composite {

	public static final String COLUMN_KEY_DIAGRAM = "diagram"; //$NON-NLS-1$
	public static final String COLUMN_KEY_DOC = "doc"; //$NON-NLS-1$

	/**
	 * This exists to make {@link NodeCheckStateProvider} faster
	 */
	private static class CheckStateCache {
		Map<Node, Boolean> isChecked = new HashMap<>();
		Map<Node, Boolean> isGrayed = new HashMap<>();
		Map<Node, Boolean> hasPrintablesDeep = new HashMap<>();

		public void invalidate(Node n) {
			for (; n != null; n = n.getParent()) {
				isChecked.remove(n);
				isGrayed.remove(n);
				hasPrintablesDeep.remove(n);
			}
		}

		public void invalidate() {
			isChecked.clear();
			isGrayed.clear();
			hasPrintablesDeep.clear();
		}
	}

	protected Display              display;

	protected LocalResourceManager resourceManager;

	protected Color                noContentColor;
	protected Color                availableColor;
	protected Color                includedColor;

	protected IFilterStrategy      filterStrategy     = new DefaultFilterStrategy();

	protected Text                 filter;

	protected Matcher              matcher            = null;

	protected CheckboxTreeViewer   tree;

	protected Button               selectSubtreeToggle;

	/**
	 * The tree paths that were expanded last time no filter was defined. Will
	 * be nullified after the expanded paths have been returned when
	 * {@link #matcher} turns null.
	 */
	protected TreePath[]           noFilterExpandedPaths;

	protected Set<Node>            selectedNodes;

	protected CheckStateCache      checkStateCache = new CheckStateCache();

	protected Runnable             selectionChangeListener;

	protected CollectionResult     nodes;

	private BiPredicate<Node, String> inclusionFilter = (n, k) -> true;

	private transient boolean selectSubtreeEnabled = true;

	public NodeTree(Composite parent, Set<Node> selectedNodes) {
		super(parent, 0);

		this.display = getDisplay();
		this.selectedNodes = selectedNodes;

		resourceManager = new LocalResourceManager(JFaceResources.getResources(), this);
		noContentColor = getDisplay().getSystemColor(SWT.COLOR_DARK_GRAY);
		availableColor = getDisplay().getSystemColor(SWT.COLOR_RED);
		includedColor = getDisplay().getSystemColor(SWT.COLOR_BLUE);

		GridLayoutFactory.fillDefaults().spacing(20, 10).numColumns(3).applyTo(this);

		createFilter(this);
		createTree(this);
		createButtons(this);
	}

	public void setInclusionFilter(BiPredicate<Node, String> f) {
		this.inclusionFilter = f != null ? f : (n, k) -> true;
	}

	public void setSelectionChangeListener(Runnable r) {
		this.selectionChangeListener = r;
	}

	public void setInput(CollectionResult nodes) {
		this.nodes = nodes;

//		System.out.println("NODE TREE INPUT:\n");
//		for (Node root : nodes.roots)
//			root.print(System.out);

		tree.setInput(nodes);
		resetFilterString(filter.getText());
	}

	private Runnable resetFilter = () -> resetFilterString(filter.getText());

	private void createFilter(Composite parent) {
		Label filterLabel = new Label(parent, SWT.NONE);
		filterLabel.setText(Messages.NodeTree_Filter);
		GridDataFactory.fillDefaults().span(1, 1).applyTo(filterLabel);
		filter = new Text(parent, SWT.BORDER);
		GridDataFactory.fillDefaults().span(2, 1).applyTo(filter);
		filter.addModifyListener(e -> display.timerExec(500, resetFilter));
	}

	private void createTree(Composite parent) {
		Composite treeParent = new Composite(parent, SWT.NONE);
		GridDataFactory.fillDefaults().grab(true, true).span(3, 1).applyTo(treeParent);
		TreeColumnLayout treeLayout = new TreeColumnLayout();
		treeParent.setLayout(treeLayout);

		tree = new CheckboxTreeViewer(treeParent, SWT.BORDER | SWT.MULTI | SWT.FULL_SELECTION);
		tree.getTree().setLinesVisible(true);
		tree.getTree().setHeaderVisible(true);
		tree.setUseHashlookup(true);
		//GridDataFactory.fillDefaults().grab(true, true).span(3, 1).applyTo(tree.getControl());
		tree.getControl().setToolTipText(Messages.NodeTree_Tooltip);
		tree.setAutoExpandLevel(2);
		tree.addCheckStateListener(new CheckStateListener());
		tree.setContentProvider(new NodeTreeContentProvider());
		tree.setLabelProvider(new NodeLabelProvider());
		tree.setCheckStateProvider(new NodeCheckStateProvider());
		tree.setComparator(new ViewerComparator(AlphanumComparator.CASE_INSENSITIVE_COMPARATOR));
		tree.setFilters(new ViewerFilter[] { new NodeFilter() });

		TreeViewerColumn c1 = new TreeViewerColumn(tree, SWT.LEFT);
		c1.getColumn().setText(Messages.NodeTree_Column_Item);
		c1.getColumn().setWidth(200);
		c1.setLabelProvider(new NodeLabelProvider());
		treeLayout.setColumnData(c1.getColumn(), new ColumnWeightData(100, 200));

		TreeViewerColumn c2 = new TreeViewerColumn(tree, SWT.CENTER);
		c2.getColumn().setText(Messages.NodeTree_Column_Diagram);
		c2.getColumn().setWidth(130);
		c2.setLabelProvider(new HasDiagramLabelProvider());
		treeLayout.setColumnData(c2.getColumn(), new ColumnWeightData(0, 130));

		TreeViewerColumn c3 = new TreeViewerColumn(tree, SWT.CENTER);
		c3.getColumn().setText(Messages.NodeTree_Column_Documentation);
		c3.getColumn().setWidth(130);
		c3.setLabelProvider(new HasDocLabelProvider());
		treeLayout.setColumnData(c3.getColumn(), new ColumnWeightData(0, 130));
	}

	private void createButtons(Composite parent) {
		Composite bar = new Composite(parent, SWT.NONE);
		GridDataFactory.fillDefaults().grab(true, false).span(3, 1).applyTo(bar);
		bar.setLayout(new RowLayout());

		selectSubtreeToggle = new Button(bar, SWT.TOGGLE);
		selectSubtreeToggle.setText(Messages.NodeTree_SubtreeSelection_Text);
		selectSubtreeToggle.setToolTipText(subtreeToggleTooltip(false));
		selectSubtreeToggle.setSelection(selectSubtreeEnabled);
		selectSubtreeToggle.addSelectionListener(SelectionListener.widgetSelectedAdapter(e -> {
			selectSubtreeEnabled = selectSubtreeToggle.getSelection();
			// Check state cache must be invalidated!
			refreshTree(true);
		}));

		Button expand = new Button(bar, SWT.PUSH);
		expand.setText(Messages.NodeTree_Expand_Text);
		expand.setToolTipText(Messages.NodeTree_Expand_Tooltip);
		expand.addSelectionListener(SelectionListener.widgetSelectedAdapter(e -> {
			IStructuredSelection ss = tree.getStructuredSelection();
			if (ss.isEmpty())
				tree.expandAll();
			else
				for (Object n : ss.toList())
					tree.expandToLevel(n, TreeViewer.ALL_LEVELS);
			scheduleFocusTree();
		}));
		Button collapse = new Button(bar, SWT.PUSH);
		collapse.setText(Messages.NodeTree_Collapse_Text);
		collapse.setToolTipText(Messages.NodeTree_Collapse_Tooltip);
		collapse.addSelectionListener(SelectionListener.widgetSelectedAdapter(e -> {
			IStructuredSelection ss = tree.getStructuredSelection();
			if (ss.isEmpty())
				tree.collapseAll();
			else
				for (Object n : ss.toList())
					tree.collapseToLevel(n, TreeViewer.ALL_LEVELS);
			scheduleFocusTree();
		}));
	}

	protected void fireChangeListener() {
		if (selectionChangeListener != null)
			selectionChangeListener.run();
	}

	protected void scheduleFocusTree() {
		display.asyncExec(() -> {
			if (!tree.getTree().isDisposed() && !tree.getTree().isFocusControl())
				tree.getTree().setFocus();
		});
	}

	private Collection<Node> getVisibleNodes() {
		Collection<Node> result = new ArrayList<>();

		Deque<TreeItem> todo = new ArrayDeque<>();
		for (TreeItem ti : tree.getTree().getItems()) {
			todo.add(ti);
		}

		while (!todo.isEmpty()) {
			TreeItem item = todo.removeLast();
			Node node = (Node) item.getData();
			if (node != null)
				result.add(node);

			for (TreeItem child : item.getItems()) {
				todo.add(child);
			}
		}

		return result;
	}

	private void resetFilterString(String filterString) {
		TreePath[] restoreExpansions = null;
		String patternString = filterStrategy.toPatternString(filterString);
		if (patternString == null) {
			if (matcher != null) {
				// Filter has been removed
				restoreExpansions = noFilterExpandedPaths;
				noFilterExpandedPaths = null;
			}
			matcher = null;
		} else {
			if (matcher == null) {
				// Filter has been defined after not being previously defined
				noFilterExpandedPaths = tree.getExpandedTreePaths();
			}
			matcher = Pattern.compile(patternString).matcher(""); //$NON-NLS-1$
		}
		selectSubtreeToggle.setToolTipText(subtreeToggleTooltip(matcher != null));
		refreshTree(false);
		if (restoreExpansions != null)
			tree.setExpandedTreePaths(restoreExpansions);
		else
			tree.expandAll();
	}

	protected static boolean hasPrintables(Node n) {
		return Nodes.HAS_PRINTABLES_PREDICATE.test(n);
	}

	protected boolean hasPrintablesDeep0(Node n) {
		if (hasPrintables(n))
			return true;
		for (Node c : n.getChildren())
			if (hasPrintablesDeep(c))
				return true;
		return false;
	}

	protected boolean hasPrintablesDeep(Node n) {
		Boolean r = checkStateCache.hasPrintablesDeep.get(n);
		if (r != null)
			return r;

		r = hasPrintablesDeep0(n);
		checkStateCache.hasPrintablesDeep.put(n, r);
		return r;
	}

	protected boolean isSomethingSelected(Node node) {
		if (hasPrintables(node) && selectedNodes.contains(node)) {
			return true;
		}

		Collection<Node> children = node.getChildren();
		if (!children.isEmpty()) {
			for (Node child : children) {
				if (!hasPrintablesDeep(child))
					continue;
				if (isSomethingSelected(child))
					return true;
			}
		}
		return false;
	}

	protected boolean isFullySelected(Node node) {
		boolean hp = hasPrintables(node);
		boolean thisSelected = selectedNodes.contains(node);
		if (hp && !thisSelected) {
			return false;
		}

		Collection<Node> children = node.getChildren();
		if (children.isEmpty()) {
			return hp && thisSelected;
		}

		int selectedCount = 0;
		int printableCount = 0;
		boolean allChildrenSelected = true;
		for (Node child : children) {
			if (!hasPrintablesDeep(child))
				continue;
			++printableCount;
			boolean childSelected = isFullySelected(child);
			allChildrenSelected &= childSelected;
			selectedCount += childSelected ? 1 : 0;
			if (!childSelected)
				break;
		}
		return allChildrenSelected
				&& (printableCount == 0 || (selectedCount > 0 && printableCount > 0));
	}

	protected boolean isPartiallySelected(Node node) {
		return isSomethingSelected(node) && !isFullySelected(node);
	}

	protected void refreshTree(boolean invalidateCheckStateCache) {
		if (invalidateCheckStateCache)
			checkStateCache.invalidate();
		tree.refresh();
	}

	public void refreshTree() {
		refreshTree(true);
	}

	public boolean addOrRemoveSelection(Node node, boolean add) {
		boolean changed = false;
		if (hasPrintables(node)) {
			if (add)
				changed = selectedNodes.add(node);
			else
				changed = selectedNodes.remove(node);
			if (changed)
				checkStateCache.invalidate(node);
		}
		return changed;
	}

	public boolean addOrRemoveSelectionRec(Node node, boolean add, Predicate<Node> filter) {
		boolean changed = false;
		if (filter == null || filter.test(node)) {
			changed |= addOrRemoveSelection(node, add);
		}
		for (Node child : node.getChildren())
			changed |= addOrRemoveSelectionRec(child, add, filter);
		return changed;
	}

	private class NodeTreeContentProvider implements ITreeContentProvider {
		@Override
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}
		@Override
		public void dispose() {
		}
		@Override
		public Object[] getElements(Object inputElement) {
			if (inputElement instanceof CollectionResult)
				return ((CollectionResult) inputElement).roots.toArray();
			return new Object[0];
		}
		@Override
		public boolean hasChildren(Object element) {
			Node n = (Node) element;
			Collection<Node> children = n.getChildren();
			if (children.isEmpty())	
				return false;
			for (Node c : children)
				if (hasPrintablesDeep(c))
					return true;
			return false;

		}
		@Override
		public Object getParent(Object element) {
			return ((Node) element).getParent();
		}
		@Override
		public Object[] getChildren(Object parentElement) {
			Node n = (Node) parentElement;
			List<Object> result = new ArrayList<>( n.getChildren().size() );
			for (Node c : n.getChildren()) 
				if (hasPrintablesDeep(c)) 
					result.add(c);
			return result.toArray();
		}
	}

	private class NodeLabelProvider extends CellLabelProvider {
		@Override
		public void update(ViewerCell cell) {
			Object e = cell.getElement();
			if (e instanceof Node) {
				Node n = (Node) e;
				String name = DiagramPrinter.formDiagramName(n, false);
				cell.setText(name);

				if (!Nodes.HAS_PRINTABLES_PREDICATE.test(n))
					cell.setForeground(noContentColor);
				else
					cell.setForeground(null);
			}
		}
	}

	private class HasDiagramLabelProvider extends CellLabelProvider {
		@Override
		public void update(ViewerCell cell) {
			Object e = cell.getElement();
			if (e instanceof Node) {
				Node n = (Node) e;
				if (n.getDiagramResource() != null) {
					if (selectedNodes.contains(n) && inclusionFilter.test(n, COLUMN_KEY_DIAGRAM)) {
						cell.setText(Messages.NodeTree_Included);
						cell.setForeground(includedColor);
					} else {
						cell.setText(Messages.NodeTree_Available);
						cell.setForeground(availableColor);
					}
				} else {
					cell.setText(Messages.NodeTree_NoContent);
					cell.setForeground(noContentColor);
				}
			}
		}
	}

	private class HasDocLabelProvider extends CellLabelProvider {
		@Override
		public void update(ViewerCell cell) {
			Object e = cell.getElement();
			if (e instanceof Node) {
				Node n = (Node) e;
				if (n.hasProperty(Node.PROP_DOC_RESOURCES)) {
					if (selectedNodes.contains(n) && inclusionFilter.test(n, COLUMN_KEY_DOC)) {
						cell.setText(Messages.NodeTree_Included);
						cell.setForeground(includedColor);
					} else {
						cell.setText(Messages.NodeTree_Available);
						cell.setForeground(availableColor);
					}
				} else {
					cell.setText(Messages.NodeTree_NoContent);
					cell.setForeground(noContentColor);
				}
			}
		}
	}

	private class CheckStateListener implements ICheckStateListener {
		@Override
		public void checkStateChanged(CheckStateChangedEvent event) {
			boolean checked = event.getChecked();
			Node checkedNode = (Node) event.getElement();

			// If node is grayed, checked will be false.
			// However, we want it to be true if the checked node has printables.
			if (!checked && hasPrintables(checkedNode)) {
				if (checkStateCache.isGrayed.getOrDefault(checkedNode, Boolean.FALSE)) {
					checked = true;
				}
			}

			Set<Node> nodes = new HashSet<>();
			Set<Node> selection = ISelectionUtils.filterSetSelection(tree.getSelection(), Node.class);
			if (selection.contains(checkedNode))
				nodes.addAll(selection);
			else
				tree.setSelection(StructuredSelection.EMPTY);
			nodes.add(checkedNode);

			boolean treeIsFiltered = !filter.getText().isEmpty();
			Set<Node> visible = treeIsFiltered && selectSubtreeEnabled ? new HashSet<>(getVisibleNodes()) : null;
			Predicate<Node> filter = visible != null ? visible::contains : null;

			for (Node node : nodes) {
				//System.out.println("check(" + node + "): checked=" + checked);
				if (selectSubtreeEnabled) {
					addOrRemoveSelectionRec(node, checked, filter);
				} else {
					addOrRemoveSelection(node, checked);
				}
			}

			tree.refresh();
			fireChangeListener();
		}
	}

	private class NodeCheckStateProvider implements ICheckStateProvider {
		@Override
		public boolean isChecked(Object element) {
			Node n = (Node) element;
			Boolean cache = checkStateCache.isChecked.get(n);
			if (cache != null)
				return cache;

			boolean checked = false;

			//System.out.println("START isChecked(subtreeSelection=" + selectSubtreeEnabled + ", " + n + ")");

			if (selectSubtreeEnabled) {
				// Checked / Grayed enabled.
				// Checked value is based on having something selected in this node or its subnodes.
				checked = isSomethingSelected(n);
			} else {
				// Only Checked enabled, Grayed is not used in this mode.
				// Checked value is based only on this node being selected or not.
				checked = selectedNodes.contains(n);
			}

			//System.out.println("END isChecked(subtreeSelection=" + selectSubtreeEnabled + ", " + n + ") = " + checked);
			checkStateCache.isChecked.put(n, checked);
			return checked;
		}
		@Override
		public boolean isGrayed(Object element) {
			Node n = (Node) element;
			Boolean cache = checkStateCache.isGrayed.get(n);
			if (cache != null)
				return cache;

			boolean grayed = false;

			//System.out.println("START isGrayed(subtreeSelection=" + selectSubtreeEnabled + ", " + n + ")");

			if (selectSubtreeEnabled) {
				// Grayed if this node and its substructure is only partially selected.
				grayed = isPartiallySelected(n);
			} else {
				// Grayed disabled.
			}

			//System.out.println("END isGrayed(subtreeSelection=" + selectSubtreeEnabled + ", " + n + ") = " + grayed);
			checkStateCache.isGrayed.put(n, grayed);
			return grayed;
		}
	}

	private class NodeFilter extends ViewerFilter {
		@Override
		public boolean select(Viewer viewer, Object parentElement, Object element) {
			if (matcher == null)
				return true;

			Node node = (Node) element;
			boolean matches = matcher.reset(node.getName().toLowerCase()).matches();
			if (matches)
				return true;

			// If any children are in sight, show this element.
			for (Node child : node.getChildren())
				if (select(viewer, element, child))
					return true;

			return false;
		}
	}

	public TreePath[] getExpandedPaths() {
		return tree.getExpandedTreePaths();
	}

	public void setExpandedPaths(TreePath[] expanded) {
		tree.setExpandedTreePaths(expanded);
	}

	public void withRedrawDisabled(Runnable r) {
		var t = tree.getTree();
		t.setRedraw(false);
		try {
			r.run();
		} finally {
			t.setRedraw(true);
		}
	}

	private static String subtreeToggleTooltip(boolean hasFilter) {
		return hasFilter
				? Messages.NodeTree_SubtreeSelection_Tooltip_Filtered
				: Messages.NodeTree_SubtreeSelection_Tooltip;
	}
}