/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.modelBrowser2.label;

import org.eclipse.jface.resource.ColorDescriptor;
import org.eclipse.jface.resource.FontDescriptor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.RGB;
import org.simantics.browsing.ui.content.LabelDecorator;
import org.simantics.browsing.ui.model.labeldecorators.LabelDecorationRule;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.Databoard;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.type.BooleanType;
import org.simantics.databoard.type.Datatype;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.PossibleModel;
import org.simantics.db.layer0.request.PossibleVariableValue;
import org.simantics.db.layer0.variable.RVI;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.subscription.ModelContexts;
import org.simantics.modeling.subscription.ModelContextsRequest;
import org.simantics.modeling.subscription.SubscriptionItemLabel;

/**
 * @author Tuukka Lehtonen
 */
public class SubscriptionItemLabelDecorationRule implements LabelDecorationRule {

    public static final SubscriptionItemLabelDecorationRule INSTANCE = new SubscriptionItemLabelDecorationRule();

    private static final ColorDescriptor RED = ColorDescriptor.createFrom(new RGB(255, 0, 0));
    private static final ColorDescriptor INVALID = ColorDescriptor.createFrom(new RGB(255, 128, 128));
    private static final ColorDescriptor GREEN  = ColorDescriptor.createFrom(new RGB(0, 160, 0));

    static LabelDecorator RED_DECO = new LabelDecorator.Stub() {
        @SuppressWarnings("unchecked")
        public <C> C decorateForeground(C color, String column, int itemIndex) {
            return (C) RED;
        }
    };

    static LabelDecorator GREEN_DECO = new LabelDecorator.Stub() {
        @SuppressWarnings("unchecked")
        public <C> C decorateForeground(C color, String column, int itemIndex) {
            return (C) GREEN;
        }
    };

    static LabelDecorator UNRESOLVABLE = new LabelDecorator.Stub() {
        @SuppressWarnings("unchecked")
        public <C> C decorateForeground(C color, String column, int itemIndex) {
            return (C) INVALID;
        }
        @SuppressWarnings("unchecked")
        public <F> F decorateFont(F font, String column, int itemIndex) {
            return (F) ((FontDescriptor) font).withStyle(SWT.ITALIC);
        }
    };

    @Override
    public boolean isCompatible(Class<?> contentType) {
        return contentType.equals(Resource.class);
    }

    @Override
    public LabelDecorator getLabelDecorator(ReadGraph graph, Object content) throws DatabaseException {
        if (!SubscriptionItemLabel.showValuesInLabel(graph))
            // Keep subscription item labeling as light as possible unless otherwise desired
            return null;

        Resource item = (Resource) content;
        ModelingResources MOD = ModelingResources.getInstance(graph);

        Binding rviBinding = graph.getService(Databoard.class).getBindingUnchecked( RVI.class );
        RVI rvi = graph.getPossibleRelatedValue(item, MOD.Subscription_Item_VariableId, rviBinding);
        if (rvi == null)
            return null;

        Resource model = graph.sync( new PossibleModel(item) );
        if (model == null)
            return null;

        ModelContexts contexts = graph.syncRequest(new ModelContextsRequest(model));
        if (contexts.getExperimentContext() == null)
            return null;

        Variable variable = rvi.resolvePossible(graph, contexts.getExperimentContext());
        if (variable == null)
            return UNRESOLVABLE;

        Datatype datatype = graph.getPossibleRelatedValue(item, MOD.Subscription_Item_Datatype, Bindings.getBindingUnchecked(Datatype.class));
        if (!(datatype instanceof BooleanType))
            return null;

        Object value = graph.sync( new PossibleVariableValue<Object>(variable) );
        if (!(value instanceof Boolean))
            return null;

        Boolean v = (Boolean) value;
        return v ? GREEN_DECO : RED_DECO;
    }

}
