/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.features;

import org.eclipse.ui.IEditorInput;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.ResourceArray;
import org.simantics.db.common.request.PossibleObjectWithType;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.project.IProject;
import org.simantics.simulation.ontology.SimulationResource;
import org.simantics.structural.stubs.StructuralResource2;
import org.simantics.ui.workbench.IEditorNamingService;
import org.simantics.ui.workbench.IResourceEditorInput;

public class EditorNamingService implements IEditorNamingService {

    public String getName(ReadGraph g, String editorId, IEditorInput in) throws DatabaseException {
        return truncated(getNameImpl(g, editorId, in), 256);
    }

    protected String getNameImpl(ReadGraph g, String editorId, IEditorInput in) throws DatabaseException {
        if (!(in instanceof IResourceEditorInput))
            return "Unsupported input: " + in;
        IResourceEditorInput input = (IResourceEditorInput) in;

        ResourceArray a = input.getResourceArray();
        if (a.isEmpty())
            return "(empty input)";

        Layer0 l0 = Layer0.getInstance(g);
        StructuralResource2 sr = StructuralResource2.getInstance(g);
        DiagramResource dr = DiagramResource.getInstance(g);
        ModelingResources mr = ModelingResources.getInstance(g);

        //System.out.println("GETNAME: " + input);
        //for (Resource r : a.resources)
        //    System.out.println("    " + GraphUtils.getReadableName(g, r));

        if (a.resources.length > 1) {
            // This is a structural input, there is a component path.
            StringBuilder sb = new StringBuilder();

            @SuppressWarnings("unused")
            int level = 0;

            Resource component = a.resources[1];
            sb.append(g.adapt(component, String.class));

            sb.append(" (");

            Resource last = a.resources[a.resources.length - 1];
            String partOfModel = null;
            Resource topmostComposite = g.getPossibleObject(last, l0.PartOf);
            if (topmostComposite != null) {
                //System.out.println("  PART OF: " + GraphUtils.getReadableName(g, topmostComposite));
                partOfModel = getModelName(g, topmostComposite);
                if (partOfModel != null) {
                    sb.append(partOfModel);
                    ++level;
                }
            }
            //System.out.println("part of model: " + partOfModel);

            if (a.resources.length > 2) {
                for (int i = a.resources.length - 1; i >= 2; --i) {
                    sb.append('.');
                    String pathElementName = g.adapt(a.resources[i], String.class);
                    //System.out.println("name for path element " + i + ": " + a.resources[i] + ": " + pathElementName);
                    sb.append(pathElementName);
                    ++level;
                }
            }

            sb.append(')');
//            sb.append(" [");
//            sb.append(level);
//            sb.append(']');
            return sb.toString();
        } else {
            // There is no structural path in the input, not a sub-component.
            Resource mainInput = a.resources[0];
            Resource composite = null;
            if (g.isInstanceOf(mainInput, dr.Diagram)) {

                composite = g.getPossibleObject(mainInput, mr.DiagramToComposite);
                if(composite == null) {
                	
                	Resource symbol = g.sync(new PossibleObjectWithType(mainInput, sr.Defines, dr.ElementClass));
                	if(symbol != null) {
                		Resource type = g.getSingleObject(symbol, l0.PartOf);
                		return g.adapt(type, String.class) + " (Symbol)";
                	}
                	
                    return g.adapt(input.getResource(), String.class);
                    
                }

                // Model configuration?
                String modelName = getModelName(g, composite);
                if (modelName != null) {
                    return modelName;
                }

                Resource componentType = g.getPossibleObject(composite, sr.Defines);
                if (componentType != null && g.isInstanceOf(componentType, sr.ComponentType)) {
                    return g.adapt(getPossibleSingletonInstance(g, componentType), String.class);
                }
                
                
                if(composite != null) {
                    String name = g.getPossibleAdapter(composite, String.class);
                    if(name != null) return name;
                }

            }

            // Symbol?
            if (g.isInstanceOf(mainInput, dr.Composite)) {
                Resource defines = g.getPossibleObject(mainInput, sr.Defines);
                if (defines != null && g.isInheritedFrom(defines, dr.DefinedElement)) {
                    Resource componentType = g.getPossibleObject(defines, mr.SymbolToComponentType);
                    if (componentType != null) {
                        return g.adapt(getPossibleSingletonInstance(g, componentType), String.class) + " (Symbol)";
                    }
                }
            }
            
        }

        String name = g.getPossibleAdapter(input.getResource(), String.class);
        if (name == null)
            name = "(no name)";
        return name;
    }

    protected Resource getPossibleSingletonInstance(ReadGraph g, Resource componentType) throws DatabaseException {
        Resource instance = null;
        return instance != null ? instance : componentType;
    }

    protected String getModelName(ReadGraph g, Resource configurationComposite) throws DatabaseException {
        // Model configuration?
        Resource model = g.getPossibleObject(configurationComposite, SimulationResource.getInstance(g).IsConfigurationOf);
        return model != null ? g.adapt(model, String.class) : null;
    }

    protected String truncated(String name, int maxLength) throws DatabaseException {
        if (name.length() <= maxLength)
            return name;
        return name.substring(0, Math.max(0, maxLength - 3)) + "...";
    }

    public static EditorNamingService install(IProject project) {
 
    	EditorNamingService service = new EditorNamingService();
        project.setHint(IEditorNamingService.KEY_EDITOR_NAMING_SERVICE, service);
        return service;

    }

}