/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.diagramEditor;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.UnaryRead;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.synchronization.graph.DiagramGraphUtil;
import org.simantics.g2d.diagram.participant.pointertool.TerminalUtil.TerminalInfo;
import org.simantics.g2d.element.ElementUtils;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.ui.diagramEditor.TerminalInformer.TerminalNamingStrategy;
import org.simantics.structural.stubs.StructuralResource2;
import org.simantics.structural2.queries.Terminal;

/**
 * @author Tuukka Lehtonen
 */
public final class DefaultTerminalNamingStrategy implements TerminalNamingStrategy {

    @Override
    public String getName(ReadGraph graph, TerminalInfo info) throws DatabaseException {
        Terminal t = graph.syncRequest(new ResolveTerminal(info));
        if (t == null)
            return "Could not resolve component terminal"; //$NON-NLS-1$
        return graph.syncRequest(new TerminalMessage(t));
    }

    public static class ResolveTerminal extends UnaryRead<TerminalInfo, Terminal> {

        public ResolveTerminal(TerminalInfo parameter) {
            super(parameter);
        }

        @Override
        public Terminal perform(ReadGraph graph) throws DatabaseException {
            ModelingResources mr = ModelingResources.getInstance(graph);

            Object elementObject = ElementUtils.getObject(parameter.e);
            if (!(elementObject instanceof Resource))
                return null;
            Resource element = (Resource) elementObject;

            Resource diagramBindingRelation = DiagramGraphUtil.getConnectionPointOfTerminal(graph, parameter.t);
            if (diagramBindingRelation == null)
                return null;

            Resource component = graph.getPossibleObject(element, mr.ElementToComponent);
            Resource connectionRelation = graph.getPossibleObject(diagramBindingRelation, mr.DiagramConnectionRelationToConnectionRelation);
            if (component == null || connectionRelation == null)
                return null;

            return new Terminal(component, connectionRelation);
        }

    }

    static class TerminalMessage extends UnaryRead<Terminal, String> {

        public TerminalMessage(Terminal terminal) {
            super(terminal);
        }

        @Override
        public String perform(ReadGraph graph) throws DatabaseException {
            Layer0 L0 = Layer0.getInstance(graph);
            StructuralResource2 STR = StructuralResource2.getInstance(graph);

            String name = getStringPropertyValue(graph, parameter.getRelation(), L0.HasName);
            String label = getStringPropertyValue(graph, parameter.getRelation(), L0.HasLabel);
            String componentName = NameUtils.getSafeName(graph, parameter.getComponent());
            Resource componentType = graph.getPossibleType(parameter.getComponent(), STR.Component);
            String componentTypeName = componentType != null
                    ? getStringPropertyValue(graph, componentType, L0.HasLabel)
                    : null;

            StringBuilder sb = new StringBuilder();
            if (name.equals(label)) {
                sb.append(name);
            } else {
                sb.append("(").append(name).append(") ").append(label); //$NON-NLS-1$ //$NON-NLS-2$
            }
            sb.append(" [").append(componentName); //$NON-NLS-1$
            if (componentTypeName != null)
                sb.append(" : ").append(componentTypeName); //$NON-NLS-1$
            sb.append("]"); //$NON-NLS-1$

            return sb.toString();
        }

        private String getStringPropertyValue(ReadGraph graph, Resource entity, Resource property) throws DatabaseException {
            String name = graph.getPossibleRelatedValue2(entity, property);
            return name != null ? name : NameUtils.getSafeName(graph, entity);
        }

    }

}
