package org.simantics.modeling.ui.componentTypeEditor;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.operation.IRunnableContext;
import org.eclipse.jface.text.Document;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.Position;
import org.eclipse.jface.text.source.Annotation;
import org.eclipse.jface.text.source.AnnotationModel;
import org.eclipse.jface.text.source.IAnnotationModel;
import org.eclipse.ui.texteditor.AbstractDocumentProvider;
import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Statement;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.scl.compiler.errors.CompilationError;
import org.simantics.scl.compiler.errors.Failable;
import org.simantics.scl.compiler.errors.Failure;
import org.simantics.scl.compiler.errors.Locations;
import org.simantics.scl.compiler.module.Module;
import org.simantics.scl.compiler.module.repository.UpdateListener;
import org.simantics.scl.osgi.SCLOsgi;
import org.simantics.scl.runtime.SCLContext;
import org.simantics.ui.workbench.ResourceEditorInput;
import org.simantics.utils.logging.TimeLogger;

public class SCLQueryEditorDocumentProvider extends AbstractDocumentProvider {
    
    Resource resource;
    String currentText;
    boolean errorHappened;
    
    AnnotationModel annotationModel = new AnnotationModel();
    SCLQueryEditor editor;
    
    public SCLQueryEditorDocumentProvider(SCLQueryEditor editor) {
        this.editor = editor;
    }
    
    private boolean isType(ReadGraph graph) throws DatabaseException {
    	ModelingResources MOD = ModelingResources.getInstance(graph);
    	return graph.isInstanceOf(resource, MOD.SCLQueryType);
    }
    
    @Override
    protected IDocument createDocument(Object element) throws CoreException {
        ResourceEditorInput input = (ResourceEditorInput)element;
        resource = input.getResource();
        try {
            return Simantics.getSession().syncRequest(new UniqueRead<Document>() {
                @Override
                public Document perform(ReadGraph graph) throws DatabaseException {
                    Layer0 L0 = Layer0.getInstance(graph);
                    ModelingResources MOD = ModelingResources.getInstance(graph);
                    if(isType(graph)) {
                    	Collection<Resource> assertions = graph.getAssertedObjects(resource, MOD.SCLQuery_values);
                    	if(assertions.size() != 1) throw new DatabaseException("No query text assertion defined in Query Type"); //$NON-NLS-1$
                    	Resource value = assertions.iterator().next();
	                    currentText = graph.getRelatedValue(value, L0.SCLValue_expression, Bindings.STRING);
	                    errorHappened = false;
	                    return new Document(currentText != null ? currentText : ""); //$NON-NLS-1$
                    } else {
	                    Resource value = graph.getSingleObject(resource, MOD.SCLQuery_values);
	                    currentText = graph.getRelatedValue(value, L0.SCLValue_expression, Bindings.STRING);
	                    errorHappened = false;
	                    return new Document(currentText != null ? currentText : ""); //$NON-NLS-1$
                    }
                }
            });
        } catch (DatabaseException e) {
            StringWriter sw = new StringWriter();
            PrintWriter pw = new PrintWriter(sw);
            e.printStackTrace(pw);
            errorHappened = false;
            return new Document(sw.toString());
        }
    }
    
    // While this editor is active we do not want that this listener gets collected
    private UpdateListener listener = new UpdateListener() {
        @Override
        public void notifyAboutUpdate() {
            updateAnnotations();
        }
    };
    
    private void updateAnnotations() {
        Simantics.getSession().asyncRequest(new ReadRequest() {
            @Override
            public void run(ReadGraph graph) throws DatabaseException {
                String moduleName = graph.getURI(resource);
                SCLContext context = SCLContext.getCurrent();
                context.put("graph", graph); //$NON-NLS-1$
                Failable<Module> result = SCLOsgi.MODULE_REPOSITORY.getModule(moduleName, listener);
                
                if(result instanceof Failure) {
                    Failure failure = (Failure)result;
                    setAnnotations(Arrays.asList(failure.errors));
                }
                else {
                    setAnnotations(Collections.<CompilationError>emptyList());
                }
            }
        });
    }
    
    private void setAnnotations(List<CompilationError> errors) {
        synchronized(annotationModel.getLockObject()) {
            annotationModel.removeAllAnnotations();
            for(CompilationError error : errors) {
                Annotation annotation = new Annotation("org.eclipse.ui.workbench.texteditor.error", true, //$NON-NLS-1$
                        error.description);
                int begin = Locations.beginOf(error.location);
                int end = Locations.endOf(error.location);
                Position position = new Position(begin, end - begin);
                annotationModel.addAnnotation(annotation, position);
            }
        }
    }
    
    boolean annotationsInitialized = false;
    
    @Override
    protected IAnnotationModel createAnnotationModel(Object element)
            throws CoreException {
        if(!annotationsInitialized) {
            updateAnnotations();
            annotationsInitialized = true;
        }
        return annotationModel;
    }

    @Override
    protected void doSaveDocument(IProgressMonitor monitor, Object element,
            IDocument document, boolean overwrite) throws CoreException {
        TimeLogger.resetTimeAndLog("SCLModuleEditorDocumentProvider.doSaveDocument"); //$NON-NLS-1$
        currentText = document.get();
        Simantics.getSession().asyncRequest(new WriteRequest() {
        	
        	private Resource getValue(WriteGraph graph) throws DatabaseException {

                Layer0 L0 = Layer0.getInstance(graph);
                ModelingResources MOD = ModelingResources.getInstance(graph);

        		if(isType(graph)) {
	
	        		Collection<Resource> assertions = graph.getAssertedObjects(resource, MOD.SCLQuery_values);
	            	if(assertions.size() > 1) throw new DatabaseException("Invalid query text assertions in Query Type"); //$NON-NLS-1$
	            	if(assertions.size() == 1) return assertions.iterator().next();
	            	
	            	Resource value = graph.newResource();
	            	graph.claim(value, L0.InstanceOf, MOD.SCLQuery_Value);
	            	Layer0Utils.assert_(graph, resource, MOD.SCLQuery_values, value);
	            	return value;
	            	
                } else {
                	
                	Statement stm = graph.getSingleStatement(resource, MOD.SCLQuery_values);
                	if(stm.isAsserted(resource)) {

    	            	Resource value = graph.newResource();
    	            	graph.claim(value, L0.InstanceOf, MOD.SCLQuery_Value);
    	            	graph.claim(resource, MOD.SCLQuery_values, value);
    	            	return value;

                	} else {
                		return stm.getObject();
                	}
                	
                }
        		
        	}
        	
            @Override
            public void perform(WriteGraph graph) throws DatabaseException {
                graph.markUndoPoint();
                Layer0 L0 = Layer0.getInstance(graph);
                Resource value = getValue(graph);
                graph.claimLiteral(value, L0.SCLValue_expression, currentText, Bindings.STRING);
                Layer0Utils.addCommentMetadata(graph, "Saved SCL Query " + graph.getRelatedValue2(resource, Layer0.getInstance(graph).HasName, Bindings.STRING)); //$NON-NLS-1$
            }
        });
    }

    @Override
    protected IRunnableContext getOperationRunner(IProgressMonitor monitor) {
        return null;
    }
    
    @Override
    public boolean isModifiable(Object element) {
        return !errorHappened;
    }
    
    @Override
    public boolean isReadOnly(Object element) {
        return errorHappened;
    }
    
}
