/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.componentTypeEditor;

import java.util.ArrayList;

import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.resource.LocalResourceManager;
import org.eclipse.jface.resource.ResourceManager;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Sash;
import org.eclipse.ui.forms.IMessageManager;
import org.eclipse.ui.forms.widgets.Form;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.osgi.framework.BundleContext;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.type.Datatype;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.NamedResource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.Listener;
import org.simantics.db.request.Read;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ui.Activator;
import org.simantics.modeling.utils.ComponentTypePropertiesResult;
import org.simantics.modeling.utils.ComponentTypeViewerPropertyInfo;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.SWTUtils;

public class ComponentTypeViewer {

    ComponentTypeViewerData data;
    
    ResourceManager resourceManager;
    
    ArrayList<ComponentTypeViewerSection> sections = new ArrayList<ComponentTypeViewerSection>(3);

    public ComponentTypeViewer(Composite parent, Resource _componentType, String formTitle) {
        // Form
        FormToolkit tk = new FormToolkit(parent.getDisplay());
        Form form = tk.createForm(parent);
        tk.decorateFormHeading(form);
        resourceManager = new LocalResourceManager(JFaceResources.getResources(), form);
        form.setText(formTitle);
        form.setImage(resourceManager.createImage(Activator.COMPONENT_TYPE_ICON));
        {
            FormLayout layout = new FormLayout();
            layout.marginBottom = 10;
            layout.marginTop = 10;
            layout.marginLeft = 10;
            layout.marginRight = 10;
            form.getBody().setLayout(layout);
        }
        
        // Data
        data = new ComponentTypeViewerData(tk, _componentType, form);

        // Sections
        
        sections.add(new ConfigurationPropertiesSection(data));
        sections.add(new DerivedPropertiesSection(data));

        BundleContext bundleContext = Activator.getContext();
        try {
            ArrayList<ComponentTypeViewerSectionFactory> factories = 
                    Simantics.getSession().syncRequest(new Read<ArrayList<ComponentTypeViewerSectionFactory>>() {
                        @Override
                        public ArrayList<ComponentTypeViewerSectionFactory> perform(
                                ReadGraph graph) throws DatabaseException {
                            ArrayList<ComponentTypeViewerSectionFactory> factories =
                                    new ArrayList<ComponentTypeViewerSectionFactory>(3);
                            try {
                                String className = ComponentTypeViewerSectionFactory.class.getName();
                                ServiceReference<?>[] references = bundleContext.getAllServiceReferences(className, null);
                                if(references != null)
                                    for(ServiceReference<?> reference : references) {
                                        ComponentTypeViewerSectionFactory factory = (ComponentTypeViewerSectionFactory)bundleContext.getService(reference);
                                        if(factory.doesSupport(graph, _componentType))
                                            factories.add(factory);
                                    }
                            } catch (InvalidSyntaxException e) {
                                e.printStackTrace();
                            }
                            return factories;
                        }
                    });
            for(ComponentTypeViewerSectionFactory factory : factories)
                sections.add(factory.create(data));
        } catch(DatabaseException e) {
            e.printStackTrace();
        }
        
        sections.sort((a,b) -> { return Double.compare(a.getPriority(), b.getPriority()); });
        
        // Sashes
        
        Sash[] sashes = new Sash[sections.size()-1];
        for(int i=0;i<sections.size()-1;++i) {
            Sash sash = new Sash(form.getBody(), SWT.HORIZONTAL);
            sash.setBackground(sash.getDisplay().getSystemColor(SWT.COLOR_WHITE));
            {
                FormData data = new FormData();
                data.top = new FormAttachment(100*(i+1)/sections.size(), 0);
                data.left = new FormAttachment(0, 0);
                data.right = new FormAttachment(100, 0);
                data.height = 10;
                sash.setLayoutData(data);
            }
            sash.addSelectionListener(new SelectionAdapter() {
                public void widgetSelected(SelectionEvent e) {
                    sash.setBounds(e.x, e.y, e.width, e.height);
                    FormData data = new FormData();
                    data.top = new FormAttachment(0, e.y);
                    data.left = new FormAttachment(0, 0);
                    data.right = new FormAttachment(100, 0);
                    data.height = 10;
                    sash.setLayoutData(data);
                    form.getBody().layout(true);
                }
            });
            sashes[i] = sash;
        }
        for(int i=0;i<sections.size();++i) {
            {
                FormData formData = new FormData();
                formData.top = i > 0 ? new FormAttachment(sashes[i-1]) : new FormAttachment(0, 0);
                formData.left = new FormAttachment(0, 0);
                formData.right = new FormAttachment(100, 0);
                formData.bottom = i < sections.size()-1 
                        ? new FormAttachment(sashes[i]) : new FormAttachment(100, 0);
                sections.get(i).getSection().setLayoutData(formData);
            }
        }

        // Listening
        createGraphListener();
    }
    
    private void createGraphListener() {
        Simantics.getSession().asyncRequest(new ComponentTypePropertiesResultRequest(this.data.componentType, this.sections), new Listener<ComponentTypePropertiesResult>() {
            @Override
            public void execute(final ComponentTypePropertiesResult result) {  
                SWTUtils.asyncExec(data.form.getDisplay(), new Runnable() {
                    @Override
                    public void run() {
                        // Store loaded properties
                        data.properties = result.getProperties().toArray(new ComponentTypeViewerPropertyInfo[result.getProperties().size()]);
                        data.connectionPoints = result.getConnectionPoints().toArray(new NamedResource[result.getConnectionPoints().size()]);

                        for(ComponentTypeViewerSection section : sections)
                            section.update(result);
                        setReadOnly(result.isImmutable());
                    }
                });
            }

            @Override
            public void exception(Throwable t) {
                ErrorLogger.defaultLogError(t);
            }

            @Override
            public boolean isDisposed() {
                return data.form.isDisposed();
            }

        });
    }

    public void setFocus() {
        data.form.setFocus();
    }

    /**
     * @param readOnly
     * @thread SWT
     */
    private void setReadOnly(boolean readOnly) {
        if (readOnly == data.readOnly)
            return;
        data.readOnly = readOnly;
        for(ComponentTypeViewerSection section : sections)
            section.setReadOnly(readOnly);
        IMessageManager mm = data.form.getMessageManager();
        if (readOnly)
            mm.addMessage("readonly", Messages.ComponentTypeViewer_OpenedInReadOnly, null, IMessageProvider.INFORMATION); //$NON-NLS-1$
        else
            mm.removeMessage("readonly"); //$NON-NLS-1$
    }

}
