/*******************************************************************************
 * Copyright (c) 2007, 2023 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - #7297, gitlab #987
 *******************************************************************************/
package org.simantics.modeling.ui.pdf;

import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;
import java.util.function.BiPredicate;

import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.ControlListener;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.NamedResource;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.modeling.requests.CollectionResult;
import org.simantics.modeling.requests.Node;
import org.simantics.modeling.requests.Nodes;
import org.simantics.modeling.ui.Activator;
import org.simantics.modeling.ui.pdf.PDFExportPlan.StoredNodeSelection;
import org.simantics.ui.utils.ResourceAdaptionUtils;
import org.simantics.utils.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PDFExportPage extends WizardPage {

    private static final Logger    LOGGER = LoggerFactory.getLogger(PDFExportPage.class);

    private static final String KEY_NODE_TREE_FONT_DATA = "NodeTree.FontData"; //$NON-NLS-1$
    private static final String KEY_NODE_TREE_HEIGHT = "NodeTree.Height"; //$NON-NLS-1$
    private static final String KEY_NODE_TREE_WIDTH = "NodeTree.Width"; //$NON-NLS-1$

    private static final String DIALOG = "PDFExportPage"; //$NON-NLS-1$

    protected Display              display;

    protected PDFExportPlan        exportModel;

    protected Combo                modelSelector;
    protected SelectionListener    modelSelectorListener;

    protected NodeTree             nodeTree;

    protected CCombo               exportLocation;
    protected ModifyListener       exportLocationListener;

    protected Set<Node>            selectedNodes;

    protected Label                toFileLabel;

    protected boolean              exportLocationTouchedByUser = false;

    protected IDialogSettings      dialogSettings;
    protected Point                nodeTreeSize = new Point(-1, -1);

    private BiPredicate<Node, String> nodeTreeInclusionFilter = (n, k) -> {
        boolean hasDiagram = n.getDiagramResource() != null;
        boolean hasDoc = n.hasProperty(Node.PROP_DOC_RESOURCES);
        boolean includeDiagram = exportModel.includeDiagrams;
        boolean includeDoc = exportModel.includeDocumentation;
        switch (k) {
        case NodeTree.COLUMN_KEY_DIAGRAM: return includeDiagram && hasDiagram;
        case NodeTree.COLUMN_KEY_DOC:     return includeDoc     && hasDoc;
        }
        return false;
    };

    protected PDFExportPage(PDFExportPlan model) {
        super("Export Model Contents to PDF",
                Messages.PDFExportPage_PageTitle,
                null);
        this.exportModel = model;
        this.selectedNodes = exportModel.selectedNodeSet;

        IDialogSettings settings = Activator.getDefault().getDialogSettings();
        dialogSettings = settings.getSection(DIALOG);
        if (dialogSettings == null)
            dialogSettings = settings.addNewSection(DIALOG);

        loadNodeTreeSize();
    }

    private void saveNodeTreeSize() {
        if (dialogSettings != null) {
            Point s = nodeTree.getSize();
            dialogSettings.put(KEY_NODE_TREE_WIDTH,  s.x); //$NON-NLS-1$
            dialogSettings.put(KEY_NODE_TREE_HEIGHT, s.y); //$NON-NLS-1$
            FontData [] fontDatas = JFaceResources.getDialogFont().getFontData();
            if (fontDatas.length > 0) {
                dialogSettings.put(KEY_NODE_TREE_FONT_DATA, fontDatas[0].toString()); //$NON-NLS-1$
            }
        }
    }

    private void loadNodeTreeSize() {
        if (dialogSettings != null) {
            boolean useStoredBounds = true;
            String previousDialogFontData = dialogSettings.get(KEY_NODE_TREE_FONT_DATA); //$NON-NLS-1$
            // There is a previously stored font, so we will check it.
            if (previousDialogFontData != null && previousDialogFontData.length() > 0) {
                FontData [] fontDatas = JFaceResources.getDialogFont().getFontData();
                if (fontDatas.length > 0) {
                    String currentDialogFontData = fontDatas[0].toString();
                    useStoredBounds = currentDialogFontData.equalsIgnoreCase(previousDialogFontData);
                }
            }
            if (useStoredBounds) {
                try {
                    Point sz = new Point(-1, -1);
                    // Get the stored width and height.
                    int width = dialogSettings.getInt(KEY_NODE_TREE_WIDTH); //$NON-NLS-1$
                    if (width != Dialog.DIALOG_DEFAULT_BOUNDS) {
                        sz.x = width;
                    }
                    int height = dialogSettings.getInt(KEY_NODE_TREE_HEIGHT); //$NON-NLS-1$
                    if (height != Dialog.DIALOG_DEFAULT_BOUNDS) {
                        sz.y = height;
                    }
                    nodeTreeSize = sz;
                } catch (NumberFormatException e) {
                }
            }
        }
    }

    @Override
    public void createControl(Composite parent) {
        this.display = parent.getDisplay();

        Composite container = new Composite(parent, SWT.NONE);
        {
            GridLayout layout = new GridLayout();
            layout.horizontalSpacing = 20;
            layout.verticalSpacing = 10;
            layout.numColumns = 3;
            container.setLayout(layout);
        }

        Label modelSelectorLabel = new Label(container, SWT.NONE);
        modelSelectorLabel.setText(Messages.PDFExportPage_ModelSelectorLabel);
        GridDataFactory.fillDefaults().span(1, 1).applyTo(modelSelectorLabel);
        modelSelector = new Combo(container, SWT.BORDER | SWT.READ_ONLY);
        GridDataFactory.fillDefaults().span(2, 1).applyTo(modelSelector);
        modelSelectorListener = new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                NamedResource data = (NamedResource) modelSelector.getData(String.valueOf(modelSelector.getSelectionIndex()));
                scheduleInitializeData(data);
            }
        };

        // Fill model selector combo
        for (int i = 0; i < exportModel.selectableModels.size(); ++i) {
            NamedResource nr = exportModel.selectableModels.get(i);
            modelSelector.add(nr.getName());
            modelSelector.setData("" + i, nr); //$NON-NLS-1$
        }

        modelSelector.addSelectionListener(modelSelectorListener);

        nodeTree = new NodeTree(container, selectedNodes);
        nodeTree.setInclusionFilter(nodeTreeInclusionFilter);

        GridDataFactory.fillDefaults()
        .minSize(600, 500)
        .hint(nodeTreeSize.x, nodeTreeSize.y)
        .grab(true, true)
        .span(3, 1)
        .applyTo(nodeTree);

        nodeTree.addControlListener(ControlListener.controlResizedAdapter(e -> {
            saveNodeTreeSize();
        }));
        nodeTree.setSelectionChangeListener(this::validatePage);

        toFileLabel = new Label(container, SWT.NONE);
        toFileLabel.setText(Messages.PDFExportPage_TargetSelectorLabel);
        exportLocation = new CCombo(container, SWT.BORDER);
        {
            exportLocation.setText(""); //$NON-NLS-1$
            GridDataFactory.fillDefaults().grab(true, false).span(1, 1).applyTo(exportLocation);

            for (String path : exportModel.recentLocations) {
                exportLocation.add(path);
            }

            exportLocationListener = new ModifyListener() {
                @Override
                public void modifyText(ModifyEvent e) {
                    //System.out.println("export location changed by user");
                    exportLocationTouchedByUser = true;
                    validatePage();
                }
            };
            exportLocation.addModifyListener(exportLocationListener);
        }
        Button browseFileButton = new Button(container, SWT.PUSH);
        {
            browseFileButton.setText(Messages.PDFExportPage_BrowseButtonText);
            browseFileButton.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, false));
            browseFileButton.addSelectionListener(new SelectionAdapter() {
                @Override
                public void widgetSelected(SelectionEvent e) {
                    FileDialog dialog = new FileDialog(getShell(), SWT.SAVE);
                    dialog.setFilterExtensions(new String[] { "*.pdf" }); //$NON-NLS-1$
                    dialog.setFilterNames(new String[] { Messages.PDFExportPage_PDFFileDialogFilterName });
                    String loc = exportLocation.getText();
                    if (loc != null) {
                        IPath p = new Path(loc);
                        File f = p.toFile();
                        if (f.isDirectory()) {
                            dialog.setFilterPath(f.toString());
                        } else if (f.isFile()) {
                            IPath path = p.removeLastSegments(1);
                            String name = p.lastSegment();
                            dialog.setFilterPath(path.toOSString());
                            dialog.setFileName(name);
                        } else {
                            dialog.setFilterPath(f.toString());
                            IPath path = p.removeLastSegments(1);
                            String name = p.lastSegment();
                            f = path.toFile();
                            if (f.isDirectory()) {
                                dialog.setFilterPath(path.toOSString());
                            }
                            dialog.setFileName(name);
                        }
                    }
                    String file = dialog.open();
                    if (file == null)
                        return;
                    exportLocation.setText(file);
                    validatePage();
                }
            });
        }

        final Button zoomToFitButton = new Button(container, SWT.CHECK);
        GridDataFactory.fillDefaults().grab(true, false).span(3, 1).applyTo(zoomToFitButton);
        zoomToFitButton.setText(Messages.PDFExportPage_FitByContent);
        zoomToFitButton.setSelection(exportModel.fitContentToPageMargins);
        zoomToFitButton.addSelectionListener(SelectionListener.widgetSelectedAdapter(e -> {
            exportModel.fitContentToPageMargins = zoomToFitButton.getSelection();
        }));

        /*
        final Button attachTGButton = new Button(container, SWT.CHECK);
        GridDataFactory.fillDefaults().grab(true, false).span(3, 1).applyTo( attachTGButton );
        attachTGButton.setText("Attach &TG (Importable diagram)");
        attachTGButton.setSelection(exportModel.attachTG);
        attachTGButton.addSelectionListener(SelectionListener.widgetSelectedAdapter(e -> {
            exportModel.attachTG = attachTGButton.getSelection();
        }));
        */

        final Button includeDiagramsButton = new Button(container, SWT.CHECK);
        GridDataFactory.fillDefaults().grab(true, false).span(3, 1).applyTo( includeDiagramsButton );
        includeDiagramsButton.setText(Messages.PDFExportPage_IncludeDiagramPages);
        includeDiagramsButton.setSelection(exportModel.includeDiagrams);
        includeDiagramsButton.addSelectionListener(SelectionListener.widgetSelectedAdapter(e -> {
            exportModel.includeDiagrams = includeDiagramsButton.getSelection();
            nodeTree.refreshTree();
            validatePage();
        }));

        final Button includeDocumentationButton = new Button(container, SWT.CHECK);
        GridDataFactory.fillDefaults().grab(true, false).span(3, 1).applyTo( includeDocumentationButton );
        includeDocumentationButton.setText(Messages.PDFExportPage_IncludeDocumentationPages);
        includeDocumentationButton.setSelection(exportModel.includeDocumentation);

        final Button includeComponentLevelDocumentationButton = new Button(container, SWT.CHECK);
        GridDataFactory.fillDefaults().grab(true, false).span(3, 1).applyTo( includeComponentLevelDocumentationButton );
        includeComponentLevelDocumentationButton.setText(Messages.PDFExportPage_IncludeComponentLevelDocumentation);
        includeComponentLevelDocumentationButton.setSelection(exportModel.includeComponentLevelDocumentation);
        includeComponentLevelDocumentationButton.setEnabled(exportModel.includeDocumentation);

        includeDocumentationButton.addSelectionListener(SelectionListener.widgetSelectedAdapter(e -> {
            exportModel.includeDocumentation = includeDocumentationButton.getSelection();
            includeComponentLevelDocumentationButton.setEnabled(exportModel.includeDocumentation);
            nodeTree.refreshTree();
            validatePage();
        }));

        includeComponentLevelDocumentationButton.addSelectionListener(SelectionListener.widgetSelectedAdapter(e -> {
            exportModel.includeComponentLevelDocumentation = includeComponentLevelDocumentationButton.getSelection();
        }));

        final Button addPageNumbers = new Button(container, SWT.CHECK);
        GridDataFactory.fillDefaults().grab(true, false).span(3, 1).applyTo( addPageNumbers );
        addPageNumbers.setText(Messages.PDFExportPage_AddPageNumbers);
        addPageNumbers.setSelection(exportModel.addPageNumbers);
        addPageNumbers.addSelectionListener(SelectionListener.widgetSelectedAdapter(e -> {
            exportModel.addPageNumbers = addPageNumbers.getSelection();
        }));

        setControl(container);
        validatePage();

        scheduleInitializeData(exportModel.selection);
    }

    private void scheduleInitializeData(final NamedResource modelSelection) {
        display.asyncExec(() -> {
            try {
                if (!nodeTree.isDisposed())
                    initializeData(modelSelection);
            } catch (DatabaseException | InterruptedException e) {
                LOGGER.error("Input data initialization failed.", e); //$NON-NLS-1$
            } catch (InvocationTargetException e) {
                LOGGER.error("Input data initialization failed.", e.getTargetException()); //$NON-NLS-1$
            }
        });
    }

    private NamedResource getSelectedModel() {
        int sel = modelSelector.getSelectionIndex();
        if (sel != -1) {
            NamedResource nr = (NamedResource) modelSelector.getData("" + sel); //$NON-NLS-1$
            return nr;
        }
        return null;
    }

    private void setExportLocationWithoutNotification(String text) {
        exportLocation.removeModifyListener(exportLocationListener);
        exportLocation.setText(text);
        exportLocation.addModifyListener(exportLocationListener);
    }

    private void initializeData(final NamedResource modelSelection) throws DatabaseException, InvocationTargetException, InterruptedException {
        Set<Node> toBeSelected = new HashSet<>();

        // Store previous model selection state
        if (exportModel.selectedModel != null) {
            exportModel.tempCache.put(exportModel.selectedModel, new StoredNodeSelection(exportModel, nodeTree.getExpandedPaths()));
        }

        // Primarily restore new model selection state from cache
        StoredNodeSelection savedState = exportModel.tempCache.get(modelSelection);
        if (savedState != null) {
            exportModel.nodes = savedState.nodes;
            toBeSelected.addAll(savedState.selectedNodeSet);
        }

        if (savedState == null && modelSelection != null) {
            // Process input selection to find the model/state selected by default.

            // This may take longer than the user wants to wait without
            // notification.

            // !PROFILE
            long time = System.nanoTime();

            getWizard().getContainer().run(true, true, monitor -> {
                try {
                    SubMonitor mon = SubMonitor.convert(monitor, Messages.PDFExportPage_SearchingForExportableContent_MonitorTask, 100);
                    exportModel.sessionContext.getSession().syncRequest(new ReadRequest() {
                        @Override
                        public void run(ReadGraph graph) throws DatabaseException {
                            CollectionResult coll = exportModel.nodes = DiagramPrinter.browse(mon.newChild(100), graph, new Resource[] { modelSelection.getResource() });
                            if (coll == null)
                                return;

                            // Decide initial selection based on exportModel.initialSelection
                            if (modelSelection.equals(exportModel.initialModelSelection)) {
                                Set<Resource> selectedResources = new HashSet<>();
                                for (Object o : exportModel.initialSelection.toList()) {
                                    Resource r = ResourceAdaptionUtils.toSingleResource(o);
                                    if (r != null)
                                        selectedResources.add(r);
                                }
                                coll.walkTree(node -> {
                                    if (Nodes.HAS_PRINTABLES_PREDICATE.test(node)) {
                                        if (Nodes.parentIsInSet(toBeSelected, node))
                                            toBeSelected.add(node);
                                        else
                                            for (Resource r : node.getDefiningResources())
                                                if (selectedResources.contains(r))
                                                    toBeSelected.add(node);
                                    }
                                    return true;
                                });
                            }

                            // Filter out any excess nodes from the tree.
                            exportModel.nodes = coll = coll.withRoots(Nodes.depthFirstFilter(Nodes.HAS_PRINTABLES_PREDICATE, coll.roots));

                            // Select all if initial selection doesn't dictate anything.
                            if (toBeSelected.isEmpty())
                                toBeSelected.addAll(coll.breadthFirstFlatten(CollectionResult.HAS_PRINTABLES_FILTER));
                        }
                    });
                } catch (DatabaseException e) {
                    throw new InvocationTargetException(e);
                } finally {
                    monitor.done();
                }
            });

            // !PROFILE
            long endTime = System.nanoTime();
            if (exportModel.nodes != null)
                LOGGER.info("Found " + exportModel.nodes.diagrams.size() + " diagrams in " + ((endTime - time)*1e-9) + " seconds."); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
        }

        // Browsing was canceled by user.
        if (exportModel.nodes == null)
            return;

        exportModel.selectedModel = modelSelection;

        // Setup selected states, select everything by default.
        selectedNodes.clear();
        selectedNodes.addAll(toBeSelected);

        // Fully refresh node tree
        nodeTree.withRedrawDisabled(() -> {
            nodeTree.setInput(exportModel.nodes);
            if (savedState != null)
                nodeTree.setExpandedPaths(savedState.expandedNodes);
        });

        modelSelector.removeSelectionListener(modelSelectorListener);
        int selectedIndex = -1;
        for (int i = 0; i < modelSelector.getItemCount(); ++i) {
            Object obj = modelSelector.getData("" + i); //$NON-NLS-1$
            if (Objects.equals(obj, modelSelection)) {
                selectedIndex = i;
            }
        }
        if (selectedIndex == -1 && modelSelector.getItemCount() > 0)
            selectedIndex = 0;
        if (selectedIndex != -1)
            modelSelector.select(selectedIndex);
        modelSelector.addSelectionListener(modelSelectorListener);

        validatePage();
    }

    void validatePage() {
        int diagramCount = 0;
        int docCount = 0;
        Node singleDiagram = null;
        for (Node n : selectedNodes) {
            if (exportModel.includeDiagrams && n.getDiagramResource() != null) {
                ++diagramCount;
                singleDiagram = n;
            }
            if (exportModel.includeDocumentation && n.hasProperty(Node.PROP_DOC_RESOURCES)) {
                ++docCount;
            }
        }

        //System.out.println("VALIDATE PAGE: " + exportLocationTouchedByUser);
        if (diagramCount == 0 && docCount == 0) {
            setMessage(Messages.PDFExportPage_Message_SelectSomeItemsToExport);
            setErrorMessage(null);
            setPageComplete(false);
            return;
        }

        if (!exportLocationTouchedByUser) {
            String generatedName = null;
            // Generate file name automatically if user hasn't touched the name manually.
            NamedResource nr = getSelectedModel();
            if (nr != null) {
                if (diagramCount == 1 && singleDiagram != null) {
                    generatedName = nr.getName() + "-" + singleDiagram.getName(); //$NON-NLS-1$
                } else {
                    generatedName = nr.getName();
                }
            }
            //System.out.println("generate name: " + generatedName);
            if (generatedName != null) {
                if (!FileUtils.isValidFileName(generatedName))
                    generatedName = (String) Bindings.STR_VARIANT.createUnchecked(Bindings.STRING, generatedName);
                String name = generatedName + ".pdf"; //$NON-NLS-1$

                abu:
                if ( !exportModel.recentLocations.isEmpty() ) {
                    for ( String loc : exportModel.recentLocations ) {
                        if ( loc.endsWith(name) && !loc.equals(name) ) {
                            name = loc;
                            break abu; 
                        }
                    }

                    String firstLine = exportModel.recentLocations.iterator().next();
                    File f = new File(firstLine);
                    File parentFile = f.getParentFile();
                    if (parentFile != null) {
                        name = new File( f.getParentFile(), name ).getAbsolutePath();
                    }
                }
                setExportLocationWithoutNotification(name);
            }
        }

        String exportLoc = exportLocation.getText();
        if (exportLoc.isEmpty()) {
            setMessage(Messages.PDFExportPage_Message_SelectExportTargetFile);
            setErrorMessage(null);
            setPageComplete(false);
            return;
        }
        File file = new File(exportLoc);
        if (file.exists()) {
            if (file.isDirectory()) {
                setErrorMessage(Messages.PDFExportPage_Error_TargetAlreadyExists_IsDir);
                setPageComplete(false);
                return;
            }
            if (!file.isFile()) {
                setErrorMessage(Messages.PDFExportPage_Error_TargetAlreadyExists_NotRegularFile);
                setPageComplete(false);
                return;
            }
        }
        exportModel.exportLocation = file;

        String msg = NLS.bind(Messages.PDFExportPage_Message_ExportInfo,
                new Object[] {diagramCount, docCount, diagramCount + docCount});

        setMessage(msg);
        setErrorMessage(null);
        setPageComplete(true);
    }

}
