/*******************************************************************************
 * Copyright (c) 2018 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.diagramEditor.dnd;

import java.util.List;
import java.util.stream.Collectors;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.simantics.NameLabelMode;
import org.simantics.NameLabelUtil;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ValidationException;
import org.simantics.db.layer0.request.IsLinkedTo;
import org.simantics.db.request.Write;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.commandlog.CommandRecording;

/**
 * @author Tuukka Lehtonen
 * @since 1.37.0
 */
public class DropSuggestions {

    public static boolean askSuggestions(Shell parent, List<DropSuggestion> suggestions) {
        return Dialog.OK == MarkupDialog.open(parent,
                "diagram.dropSuggestionDialog", //$NON-NLS-1$
                Messages.DropSuggestions_ApplySuggestions_DialogTitle,
                Messages.DropSuggestions_ApplySuggestions_DialogMsg,
                formatSuggestions(suggestions),
                SWT.ICON_QUESTION,
                SWT.NONE,
                SWT.BORDER | SWT.WRAP | SWT.V_SCROLL);
    }

    private static String formatSuggestions(List<DropSuggestion> suggestions) {
        return suggestions.stream()
                .map(DropSuggestion::toString)
                .collect(Collectors.joining("\n", "- ", "")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
    }

    public static Write performSuggestionsRequest(List<DropSuggestion> suggestions) {
        return (Write) graph -> {
            for (DropSuggestion s : suggestions) {
                s.fix(graph);
            }
        };
    }

    public static DropSuggestion linkToLibrary(ReadGraph graph, Resource linkSource, Resource linkTarget) throws DatabaseException {
        return new LinkToLibrary(
                NLS.bind(Messages.DropSuggestions_Description_LinkToLibrary,
                        NameLabelUtil.modalName(graph, linkSource, NameLabelMode.NAME_AND_LABEL),
                        NameLabelUtil.modalName(graph, linkTarget, NameLabelMode.NAME_AND_LABEL))
                , linkSource, linkTarget);
    }

    public static class LinkToLibrary implements DropSuggestion {

        private final String desc;
        private final Resource linkFrom;
        private final Resource linkTo;

        public LinkToLibrary(String desc, Resource linkFrom, Resource linkTo) {
            this.desc = desc;
            this.linkFrom = linkFrom;
            this.linkTo = linkTo;
        }

        @Override
        public void fix(WriteGraph graph) throws DatabaseException {
            // Ensure that there is no reverse link between the namespaces
            if (graph.syncRequest(new IsLinkedTo(linkFrom, linkTo))) {
                throw new ValidationException(
                        NLS.bind(Messages.DropSuggestions_Problem_CyclicDependency
                                , NameUtils.getURIOrSafeNameInternal(graph, linkTo)
                                , NameUtils.getURIOrSafeNameInternal(graph, linkFrom)));
            }
            Layer0 L0 = Layer0.getInstance(graph);
            graph.claim(linkFrom, L0.IsLinkedTo, L0.IsLinkedTo_Inverse, linkTo);
            CommandRecording.recordLinkSharedOntology(graph, linkFrom, linkTo);
        }

        @Override
        public String toString() {
            return desc;
        }

    }

}
