/*******************************************************************************
 * Copyright (c) 2010, 2024 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package org.simantics.modeling.ui.actions.e4;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.inject.Inject;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.e4.ui.di.UISynchronize;
import org.eclipse.e4.ui.model.application.ui.menu.MToolControl;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.action.ToolBarManager;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.ToolBar;
import org.eclipse.swt.widgets.ToolItem;
import org.simantics.Simantics;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.common.NodeContextBuilder;
import org.simantics.browsing.ui.model.InvalidContribution;
import org.simantics.browsing.ui.model.actions.ActionBrowseContext;
import org.simantics.browsing.ui.model.actions.IActionCategory;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.BinaryRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.Listener;
import org.simantics.modeling.ui.Activator;

/**
 * Used like this in E4 model fragments:
 * 
 * <pre>
 * <fragments xsi:type="fragment:StringModelFragment" featurename="trimContributions" parentElementId="org.eclipse.e4.legacy.ide.application">
 *  <elements xsi:type="menu:TrimContribution" elementId="modeled.trim.id" parentId="org.eclipse.ui.main.toolbar" positionInParent="after=additions">
 *    <children xsi:type="menu:ToolBar" elementId="toolbar.id">
 *      <children xsi:type="menu:ToolControl" elementId="toolcontrol.id" contributionURI="bundleclass://org.simantics.modeling.ui/org.simantics.modeling.ui.actions.GlobalModeledToolbarActions">
 *        <tags>http://www.simantics.org/Project-1.2/MainToolbarActionContext</tags>
 *      </children>
 *    </children>
 *  </elements>
 * </fragments>
 * </pre>
 * 
 * @author Antti Villberg
 * @author Jani Simomaa
 * @author Tuukka Lehtonen
 */
public class GlobalModeledToolbarActions {

    @Inject UISynchronize sync;

    private List<String> browseContexts;
    private boolean disposed = false;
    private Composite composite;

    @PostConstruct
    protected void create(Composite parent, MToolControl control) {
        /*
         * The parent shall remain the first element in the toolbar so minimize its appearance
         */
        parent.setLayout(GridLayoutFactory.fillDefaults().create());
        this.composite = parent;
        Composite ch = new Composite(parent, SWT.NONE);
        ch.setLayoutData(GridDataFactory.swtDefaults().hint(1,1).create());
        browseContexts = new ArrayList<>(control.getTags());
        Simantics.getSession().asyncRequest(
                new GetContributions(Simantics.getProjectResource(), browseContexts),
                new ContributionListener());
    }
    
    @PreDestroy
    private void dispose() {
        disposed = true;
        if (composite != null) {
            for (Control c : composite.getChildren())
                c.dispose();
            composite = null;
        }
    }

    class ContributionListener implements Listener<List<IContributionItem>>, Runnable {

        AtomicReference<List<IContributionItem>> lastResult = new AtomicReference<>();

        @Override
        public void execute(List<IContributionItem> result) {
            if (composite != null) {
                lastResult.set(result);
                sync.asyncExec(this);
            }
        }

        @Override
        public void run() {

            List<IContributionItem> result = lastResult.getAndSet(null);
            if (result == null || composite == null || composite.isDisposed())
                return;

            ToolBar tb = (ToolBar) composite.getParent();
            ToolItem[] mi = tb.getItems();
            if(mi.length == 0)
                return;

            ToolBarManager toolBarManager = new ToolBarManager(tb);
            /*
             * The first element contributed in E4 ToolControl will remain and cannot be removed easily
             */
            IContributionItem base = (IContributionItem)mi[0].getData();
            toolBarManager.add(base);
            for (IContributionItem item : result)
                toolBarManager.add(item);
            toolBarManager.update(true);

        }

        @Override
        public void exception(Throwable t) {
            Activator.getDefault().getLog().log(
                    new Status(IStatus.ERROR, Activator.PLUGIN_ID,
                            Messages.GlobalModeledToolbarActions_ActivatorGlobalModeledToolbarException,
                            t));
        }

        @Override
        public boolean isDisposed() {
            return disposed;
        }
    }

    static class GetContributions extends BinaryRead<Resource, Collection<String>, List<IContributionItem>> {
        public GetContributions(Resource from, Collection<String> browseContextURIs) {
            super(from, browseContextURIs);
        }
        @Override
        public List<IContributionItem> perform(ReadGraph graph) throws DatabaseException {
            return getContributionItems(graph, parameter, parameter2);
        }
    }

    private static Collection<Resource> getBrowseContextResources(ReadGraph graph, Collection<String> browseContextURIs) throws DatabaseException {
        List<Resource> result = new ArrayList<Resource>(browseContextURIs.size());
        for (String uri : browseContextURIs)
            result.add(graph.getResource(uri));
        return result;
    }

    private static List<IContributionItem> getContributionItems(ReadGraph graph, Resource from, Collection<String> browseContextURIs) throws DatabaseException {
        Collection<Resource> browseContexts = getBrowseContextResources(graph, browseContextURIs);
        NodeContext nodeContext = NodeContextBuilder.buildWithInput(from);
        try {
        	ActionBrowseContext browseContext = ActionBrowseContext.resolve(graph, browseContexts, nodeContext);
            Map<IActionCategory, List<Action>> result = browseContext.getActions(graph, nodeContext, Collections.singletonList(nodeContext));
            return toContributionItems(result);
        } catch (InvalidContribution e) {
            Activator.getDefault().getLog().log(
                    new Status(IStatus.ERROR, Activator.PLUGIN_ID,
                            Messages.GlobalModeledToolbarActions_ActivatorEncounteredInvalidContributionException,
                            e));
        }

        return Collections.emptyList();
    }

    private static List<IContributionItem> toContributionItems(Map<IActionCategory, List<Action>> map) {
        if (map.isEmpty())
            return Collections.emptyList();

        IActionCategory[] categories = map.keySet().toArray(new IActionCategory[map.size()]);
        Arrays.sort(categories, IActionCategory.ACTION_CATEGORY_COMPARATOR);

        ArrayList<IContributionItem> items = new ArrayList<>();
        boolean first = true;
        for (IActionCategory category : categories) {
            List<Action> actions = map.get(category);
            Collections.sort(actions, ACTION_COMPARATOR);

            if (category != null && category.isSubmenu()) {
                MenuManager manager = new MenuManager(category.getLabel());
                for (Action action : actions)
                    manager.add(new ActionContributionItem(action));
                items.add(manager);
            }
            else {
                if (first)
                    first = false;
                else
                    items.add(new Separator(category == null ? "" : category.getLabel())); //$NON-NLS-1$
                for (Action action : actions)
                    items.add(new ActionContributionItem(action));
            }
        }

        return items;
    }

    private static final Comparator<Action> ACTION_COMPARATOR = (o1, o2) -> {
        String t1 = o1.getText();
        String t2 = o2.getText();
        return t1 == null ?
                (t2 == null ? 0 : -1)
                : (t2 == null ? 1 : t1.compareTo(t2));
    };

}
