/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.actions;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ContributionItem;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.resource.LocalResourceManager;
import org.eclipse.jface.resource.ResourceManager;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.NamedResource;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.layer0.Layer0;
import org.simantics.project.IProject;
import org.simantics.simulation.experiment.IExperiment;
import org.simantics.simulation.ontology.SimulationResource;
import org.simantics.simulation.project.IExperimentManager;
import org.simantics.ui.contribution.DynamicMenuContribution;
import org.simantics.ui.utils.ResourceAdaptionUtils;

/**
 * @author Tuukka Lehtonen
 */
public class SetInitialState extends DynamicMenuContribution {

    private ResourceManager resourceManager;

    public SetInitialState() {
        resourceManager = new LocalResourceManager(JFaceResources.getResources());
    }

    @Override
    public void dispose() {
        if (resourceManager != null) {
            resourceManager.dispose();
            resourceManager = null;
        }
        super.dispose();
    }

    @Override
    protected boolean preAcceptSelection(Object[] selection) {
        return selection != null && selection.length == 1;
    }

    @Override
    protected IContributionItem[] getContributionItems(ReadGraph graph, Object[] selection) throws DatabaseException {
        if (selection.length != 1)
            return NONE;

        final Resource experiment = ResourceAdaptionUtils.adaptToResource(selection[0]);
        if (experiment == null)
            return NONE;

        Layer0 L0 = Layer0.getInstance(graph);
        SimulationResource SIMU = SimulationResource.getInstance(graph);
        if (!graph.isInstanceOf(experiment, SIMU.Experiment))
            return NONE;

        // Check that the experiment is not currently active.
        IProject project = Simantics.peekProject();
        if (project != null) {
            IExperimentManager expMan = project.getHint(IExperimentManager.KEY_EXPERIMENT_MANAGER);
            if (expMan != null) {
                for (IExperiment exp : new IExperiment[] { expMan.getActiveExperiment() }) {
                    if (exp != null && experiment.equals(exp.getResource()))
                        return NONE;
                }
            }
        }

        final List<NamedResource> states = new ArrayList<NamedResource>();
        Resource model = graph.getSingleObject(experiment, L0.PartOf);
        for (Resource obj : graph.syncRequest(new ObjectsWithType(model, L0.ConsistsOf, SIMU.State))) {
            String name = graph.adapt(obj, String.class);
            states.add(new NamedResource(name, obj));
        }

        final AtomicReference<NamedResource> currentInitialState = new AtomicReference<NamedResource>();
        Resource currentState = graph.getPossibleObject(experiment, SIMU.HasInitialState);
        if (currentState != null) {
            for (NamedResource state : states) {
                if (state.getResource().equals(currentState))
                    currentInitialState.set(state);
            }
        }

        // Sort the open with actions in ascending lexicographical order.
        Collections.sort(states);
        if (states.isEmpty())
            return NONE;

        return new IContributionItem[] {
                new ContributionItem() {
                    @Override
                    public void fill(Menu menu, int index) {
                        MenuItem setInitialState = new MenuItem(menu, SWT.CASCADE, index);
                        setInitialState.setText(Messages.SetInitialState_SetInitialState);
                        Menu subMenu = new Menu(menu);
                        setInitialState.setMenu(subMenu);

                        for (NamedResource state : states) {
                            addMenuItem(subMenu, new Adapter(experiment, state), state.equals(currentInitialState.get()));
                        }
                    }
                }
        };
    }

    private void addMenuItem(Menu subMenu, Adapter adapter, boolean selected) {
        MenuItem item = new MenuItem(subMenu, SWT.CHECK);
        item.setText(adapter.getText());
        item.setSelection(selected);
        ImageDescriptor imgDesc = adapter.getImageDescriptor();
        if (imgDesc != null)
            item.setImage(resourceManager.createImage(imgDesc));
        item.addSelectionListener(adapter);
    }

    static class Adapter extends Action implements SelectionListener {
        private final Resource experiment;
        private final NamedResource state;

        public Adapter(Resource experiment, NamedResource state) {
            super(state.getName());
            this.experiment = experiment;
            this.state = state;
        }

        @Override
        public void widgetDefaultSelected(SelectionEvent e) {
            widgetSelected(e);
        }

        @Override
        public void widgetSelected(SelectionEvent e) {
            run();
        }

        @Override
        public void run() {
            Simantics.getSession().asyncRequest(new WriteRequest() {
                @Override
                public void perform(WriteGraph wg) throws DatabaseException {
                    SimulationResource SIMU = SimulationResource.getInstance(wg);
                    wg.deny(experiment, SIMU.HasInitialState);
                    wg.claim(experiment, SIMU.HasInitialState, state.getResource());
                }
            });
        }
    }

}
