package org.simantics.modeling.scl.ontologymodule;

import java.util.Collection;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.procedure.adapter.TransientCacheListener;
import org.simantics.db.common.request.BinaryRead;
import org.simantics.db.common.request.UnaryRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.layer0.Layer0;
import org.simantics.scl.compiler.environment.Environment;
import org.simantics.scl.compiler.environment.Environments;
import org.simantics.scl.compiler.top.SCLExpressionCompilationException;
import org.simantics.scl.compiler.types.Type;
import org.simantics.scl.db.SCLCompilationRequestProcessor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SCLRelationInfoRequest extends BinaryRead<Resource, Environment, SCLRelationInfo> {
    private static final Logger LOGGER = LoggerFactory.getLogger(SCLRelationInfoRequest.class);
    
    private SCLRelationInfoRequest(Resource resource, Environment environment) {
        super(resource, environment);
    }

    @Override
    public SCLRelationInfo perform(ReadGraph graph) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        if(!graph.isSubrelationOf(parameter, L0.HasProperty))
            return null;
        
        String name = graph.getPossibleRelatedValue(parameter, L0.HasName);
        if(name == null)
            return null;
        
        String valueType = graph.getPossibleRelatedValue(parameter, L0.RequiresValueType);
        if(valueType == null) {
            Collection<Resource> rangeTypes = graph.getObjects(parameter, L0.HasRange);
            if(rangeTypes.size() != 1) {
                LOGGER.warn("Couldn't find SCLtype for {} because it has multiple range types.", graph.getURI(parameter));
                return null;
            }
            
            Resource range = rangeTypes.iterator().next();
            Collection<Resource> assertedValueTypes = graph.getAssertedObjects(range, L0.HasValueType);
            if(assertedValueTypes.size() != 1) {
                LOGGER.warn("Couldn't find SCL type for {} because its range {} has multiple asserted value types.", graph.getURI(parameter), graph.getURI(range));
                return null;
            }
            
            Resource assertedValueType = assertedValueTypes.iterator().next();
            valueType = graph.getPossibleValue(assertedValueType, Bindings.STRING);
            if(valueType == null) {
                LOGGER.warn("Couldn't find SCL type for {} because value type assertion of {} is missing a value.", graph.getURI(parameter), graph.getURI(range));
                return null;
            }
        }
        
        Type type;
        try {
            type = Environments.getType(parameter2, valueType);
        } catch (SCLExpressionCompilationException e) {
            LOGGER.warn("Couldn't parse the value type of relation {}. Definition was '{}'.", graph.getURI(parameter), valueType);
            return null;
        }
        
        return new SCLRelationInfo(type, name);
    }
    
    public static SCLRelationInfo getRelationInfo(Resource resource, Environment environment) {
        try {
            return SCLCompilationRequestProcessor.getRequestProcessor().syncRequest(new SCLRelationInfoRequest(resource, environment), TransientCacheListener.instance());
        } catch(DatabaseException e) {
            LOGGER.error("SCLRelationInfoRequest failed.", e);
            return null;
        }
    }

}
