package org.simantics.modeling.scl;

import gnu.trove.map.hash.THashMap;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.procedure.adapter.TransientCacheListener;
import org.simantics.db.common.request.IndexRoot;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.VariableRead;
import org.simantics.db.layer0.scl.AbstractExpressionCompilationContext;
import org.simantics.db.layer0.scl.AbstractExpressionCompilationRequest;
import org.simantics.db.layer0.util.RuntimeEnvironmentRequest2;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ProceduralSubstructureMapRequest;
import org.simantics.modeling.scl.CompileProceduralSCLMonitorRequest.CompilationContext;
import org.simantics.scl.compiler.common.names.Name;
import org.simantics.scl.compiler.constants.StringConstant;
import org.simantics.scl.compiler.elaboration.expressions.EApply;
import org.simantics.scl.compiler.elaboration.expressions.EConstant;
import org.simantics.scl.compiler.elaboration.expressions.ELiteral;
import org.simantics.scl.compiler.elaboration.expressions.EVariable;
import org.simantics.scl.compiler.elaboration.expressions.Expression;
import org.simantics.scl.compiler.environment.Environment;
import org.simantics.scl.compiler.runtime.RuntimeEnvironment;
import org.simantics.scl.compiler.types.Type;
import org.simantics.scl.runtime.SCLContext;
import org.simantics.scl.runtime.function.Function1;
import org.simantics.utils.datastructures.Pair;

public class CompileProceduralSCLMonitorRequest extends AbstractExpressionCompilationRequest<CompilationContext, Variable> {
    
    protected static Name BROWSE = Name.create("Simantics/Variables", "browse");
    protected static Name VALUE = Name.create("Simantics/Variables", "value");
    
    private final Resource componentType;
    private final Resource literal;
    private final Variable componentVariable;
    
    public static class CompilationContext extends AbstractExpressionCompilationContext {
        public final THashMap<String, Pair<String,Type>> propertyMap;
        
        public CompilationContext(RuntimeEnvironment runtimeEnvironment,
                THashMap<String, Pair<String,Type>> propertyMap) {
            super(runtimeEnvironment);
            this.propertyMap = propertyMap;
        }
    }
    
    public CompileProceduralSCLMonitorRequest(ReadGraph graph, Variable context)
            throws DatabaseException {
        this.componentType = context.getParent(graph).getType(graph);
        this.literal = context.getRepresents(graph);
        this.componentVariable = context.getParent(graph);
    }

    public static Object compileAndEvaluate(ReadGraph graph, Variable context) throws DatabaseException {
        SCLContext sclContext = SCLContext.getCurrent();
        Object oldGraph = sclContext.get("graph");
        try {
            Function1<Variable,Object> exp = graph.syncRequest(new CompileProceduralSCLMonitorRequest(graph, context),
                    TransientCacheListener.<Function1<Variable,Object>>instance());
            sclContext.put("graph", graph);
            return exp.apply(context.getParent(graph));
        } catch (DatabaseException e) {
            throw (DatabaseException)e;
        } catch (Throwable t) {
            throw new DatabaseException(t);
        } finally {
            sclContext.put("graph", oldGraph);
        }
    }

    @Override
    protected String getExpressionText(ReadGraph graph)
            throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        return graph.getRelatedValue(literal, L0.SCLValue_expression, Bindings.STRING);
    }

    @Override
    protected CompilationContext getCompilationContext(ReadGraph graph)
            throws DatabaseException {
        return graph.syncRequest(new VariableRead<CompilationContext>(componentVariable) {
            @Override
            public CompilationContext perform(ReadGraph graph)
                    throws DatabaseException {
                Resource indexRoot = graph.syncRequest(new IndexRoot(componentType));
                RuntimeEnvironment runtimeEnvironment = graph.syncRequest(new RuntimeEnvironmentRequest2(componentType, indexRoot));
                THashMap<String, Pair<String,Type>> propertyMap =
                        graph.sync(new ProceduralSubstructureMapRequest(componentVariable));
                return new CompilationContext(runtimeEnvironment, propertyMap);
            }
        });
    }

    @Override
    protected Type getContextVariableType() {
        return VARIABLE;
    }

    @Override
    protected Expression getVariableAccessExpression(
            ReadGraph graph,
            CompilationContext context,
            org.simantics.scl.compiler.elaboration.expressions.Variable contextVariable,
            String name) throws DatabaseException {
        Pair<String,Type> entry = context.propertyMap.get(name);
        if(entry == null)
            return null;
        Environment environment = context.runtimeEnvironment.getEnvironment();
        Expression propertyVariable = new EApply(
                new EConstant(environment.getValue(BROWSE)),
                new EVariable(contextVariable),
                new ELiteral(new StringConstant(entry.first))
                );
        return new EApply(
                new EConstant(environment.getValue(VALUE), entry.second),
                propertyVariable
                );
    }
    
    @Override
    public int hashCode() {
        return 31*componentVariable.hashCode() + literal.hashCode();
    }
    
    @Override
    public boolean equals(Object obj) {
        if(this == obj)
            return true;
        if(obj == null || obj.getClass() != getClass())
            return false;
        CompileProceduralSCLMonitorRequest other = (CompileProceduralSCLMonitorRequest)obj;
        return literal.equals(other.literal) && componentVariable.equals(other.componentVariable);
    }
}

