package org.simantics.modeling.migration;

import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.Statement;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.CommentMetadata;
import org.simantics.db.common.request.DelayedWriteRequest;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.migration.MigrationState;
import org.simantics.db.layer0.migration.MigrationStep;
import org.simantics.db.layer0.migration.MigrationUtils;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.db.layer0.util.RemoverUtil;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingUtils;
import org.simantics.utils.logging.TimeLogger;

public class LayerCleanupMigrationStep implements MigrationStep {

    @Override
    public void applyTo(final IProgressMonitor monitor, Session session, MigrationState state) throws DatabaseException {
    	
	    TimeLogger.log("LayerCleanupMigrationStep");

        final Resource indexRoot = MigrationUtils.getResource(monitor, session, state);
        if(indexRoot == null) return;

        monitor.setTaskName("Cleanup layers migration - remove layer assignments");

        session.sync(new DelayedWriteRequest() {

            @Override
            public void perform(WriteGraph graph) throws DatabaseException {

            	Layer0Utils.setDependenciesIndexingDisabled(graph, true);

            	Layer0 L0 = Layer0.getInstance(graph);
            	DiagramResource DIA = DiagramResource.getInstance(graph);

            	for(Resource model : graph.getObjects(indexRoot, L0.ConsistsOf)) {
            		
            		List<Resource> elements = ModelingUtils.searchByType(graph, model, DIA.Element); 
            		for(Resource element : Layer0Utils.sortByCluster(graph, elements)) {
            			for(Statement stm : graph.getStatements(element, DIA.IsFocusable))
            				graph.deny(stm);
            			for(Statement stm : graph.getStatements(element, DIA.IsVisible))
            				graph.deny(stm);
            		}
            		
            	}
            	
            	CommentMetadata cm = graph.getMetadata(CommentMetadata.class);
            	graph.addMetadata(cm.add("Removing deprecated layer assignments from the model."));
            	
            }
            
        });

        monitor.setTaskName("Cleanup layers migration - remove layers");
        
        session.sync(new WriteRequest() {

        	
            @Override
            public void perform(WriteGraph graph) throws DatabaseException {

            	Layer0Utils.setDependenciesIndexingDisabled(graph, true);

            	Layer0 L0 = Layer0.getInstance(graph);
            	DiagramResource DIA = DiagramResource.getInstance(graph);

            	for(Resource model : graph.getObjects(indexRoot, L0.ConsistsOf)) {
            		
            		for(Resource diagram : ModelingUtils.searchByTypeShallow(graph, model, DIA.Composite)) {
            			for(Resource layer : graph.getObjects(diagram, DIA.HasLayer)) {
            				if(!graph.isImmutable(layer))
            					RemoverUtil.remove(graph, layer);
            			}
            		}
            		
            	}
            	
            	CommentMetadata cm = graph.getMetadata(CommentMetadata.class);
            	graph.addMetadata(cm.add("Removing deprecated layers from the model."));
            	
            }

        });

	    TimeLogger.log("LayerCleanupMigrationStep done");

    }

}
