package org.simantics.modeling.mapping;

import java.util.ArrayList;
import java.util.Collection;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.NamedResource;
import org.simantics.db.common.utils.OrderedSetUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.impl.DefaultCopyHandler;
import org.simantics.db.layer0.util.TransferableGraphConfiguration2;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.structural.stubs.StructuralResource2;


/**
 * Copy handler for Component and it's Element.
 * 
 * FIXME: Requires patched ModelTransferableGraphSourceRequest
 * 
 *  in classifyPredicates(), excluded resources must be filtered. Otherwise the code fails with :Diagram.hasInverse having no URI.
 *  
 *  Resource predicate = stm.getPredicate();
 *	Resource object = stm.getObject();		
 *	if (exclusions.contains(object) || exclusions.contains(predicate))
 *		continue; 
 *
 * 
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 *
 */
public class ComponentAndElementCopyHandler extends DefaultCopyHandler{

	protected boolean includeComponent = true;
	protected boolean includeElement = true;
	
	public ComponentAndElementCopyHandler(Resource resource) {
		super(resource);
	}
	
	/**
	 * 
	 * @param resource Component, if component is included. Otherwise the Element.
	 * @param includeComponent
	 * @param includeElement
	 */
	public ComponentAndElementCopyHandler(Resource resource, boolean includeComponent, boolean includeElement) {
		super(resource);
		if (!includeComponent && !includeElement)
			throw new IllegalArgumentException();
		this.includeComponent = includeComponent;
		this.includeElement = includeElement;
	}
	
	

	
	@Override
	protected TransferableGraphConfiguration2 createConfiguration(ReadGraph graph, boolean cut) throws DatabaseException {
    	ModelingResources MR = ModelingResources.getInstance(graph);

    	ArrayList<Resource> exclusions = new ArrayList<Resource>();

    	ArrayList<NamedResource> roots = new ArrayList<NamedResource>();
    	
    	Resource component = null;
    	Resource element = null;
    	
    	if (includeComponent) {
    		component = getResource();
    		element = graph.getPossibleObject(component, MR.ComponentToElement);
    	} else {
    		element = getResource();
    		component = graph.getPossibleObject(element, MR.ElementToComponent);
    	}
    		// Include component as root
    	if (component != null) {
    		processComponent(graph, component, includeComponent, roots, exclusions);
    	}
	    	
	    if (element != null) {
	    	processElement(graph, element, includeElement, roots, exclusions);
	    }
	    
    	return TransferableGraphConfiguration2.createWithNames(graph, roots, exclusions, true, false);
	}
	
	protected void processComponent(ReadGraph graph, Resource component, boolean includeComponent, ArrayList<NamedResource> roots, ArrayList<Resource> exclusions) throws DatabaseException {
		Layer0 L0 = Layer0.getInstance(graph);
    	StructuralResource2 SR = StructuralResource2.getInstance(graph);
    	Resource resource = getResource();
		if (includeComponent) {
			String rootName = graph.getRelatedValue(resource, L0.HasName, Bindings.STRING);
			roots.add(new NamedResource(rootName, resource));
		} else {
			exclusions.add(component);
		}

		// filter connections
		for (Resource connection : graph.getObjects(resource, SR.IsConnectedTo))
			exclusions.add(connection);
	}
	
	protected void processElement(ReadGraph graph, Resource element, boolean includeElement, ArrayList<NamedResource> roots, ArrayList<Resource> exclusions) throws DatabaseException {
		Layer0 L0 = Layer0.getInstance(graph);
		DiagramResource DIA = DiagramResource.getInstance(graph);
    	StructuralResource2 SR = StructuralResource2.getInstance(graph);
		
		if (includeElement) {
			// Add element as a root.
			String name = graph.getRelatedValue(element, L0.HasName, Bindings.STRING);
			roots.add(new NamedResource(name, element));
		} else {
			exclusions.add(element);
		}
		
		// exclude diagram from copied data.
		Collection<Resource> diagram_ = OrderedSetUtils.getOwnerLists(graph, element, DIA.Diagram);
		for (Resource diagram : diagram_) {
			
			Resource inv = graph.getInverse(diagram);
			exclusions.add(diagram);
			exclusions.add(inv);
			
			// Exclude next and previous elements in the diagram
			Resource next = OrderedSetUtils.next(graph, diagram, element);
			Resource previous = OrderedSetUtils.prev(graph, diagram, element);
			if (next != null && !next.equals(element))
				exclusions.add(next);
			if (previous != null && !previous.equals(element) && !previous.equals(next))
				exclusions.add(previous);
		}
		// filter connections
    	for (Resource connection : graph.getObjects(element, SR.IsConnectedTo))
    		exclusions.add(connection);
	}
		
}
