/*******************************************************************************
 * Copyright (c) 2012, 2015 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - platform #5873: customizable instantiation logic
 *******************************************************************************/
package org.simantics.modeling.adapters;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.simantics.DatabaseJob;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.common.request.WriteResultRequest;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.CancelTransactionException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.ActionFactory;
import org.simantics.db.request.Read;
import org.simantics.db.request.WriteResult;
import org.simantics.diagram.commandlog.NewCompositeCommand;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.internal.Plugin;
import org.simantics.scl.runtime.function.Function3;
import org.simantics.structural.stubs.StructuralResource2;
import org.simantics.structural2.utils.StructuralUtils;
import org.simantics.ui.workbench.action.DefaultActions;
import org.simantics.utils.commandlog.Commands;
import org.simantics.utils.logging.TimeLogger;

public class NewCompositeActionFactory implements ActionFactory {

    Resource compositeType;
    String defaultName;

    public NewCompositeActionFactory(Resource compositeType, String defaultName) {
        this.compositeType = compositeType;
        this.defaultName = defaultName;
    }

    @Override
    public Runnable create(Object target_) {
        final Resource target = (Resource)target_;
        return new Runnable() {
            @Override
            public void run() {
                TimeLogger.resetTimeAndLog("NewCompositeActionFactory: create composite");
                Job job = new DatabaseJob("Create Diagram") {
                    @Override
                    protected IStatus run(IProgressMonitor monitor) {
                        monitor.beginTask("Create Diagram...", IProgressMonitor.UNKNOWN);
                        try {
                            Resource composite = createComposite(compositeType, defaultName, target);
                            if(Commands.isRecording() && composite != null)
                                Commands.record(new NewCompositeCommand(composite, compositeType, defaultName, target));

                            return Status.OK_STATUS;
                        } catch (CancelTransactionException e) {
                            return new Status(IStatus.INFO , Plugin.PLUGIN_ID, e.getMessage());
                        } catch (DatabaseException e) {
                            return new Status(IStatus.ERROR, Plugin.PLUGIN_ID, e.getMessage(), e);
                        } finally {
                            monitor.done();
                        }
                    }
                };
                //job.setUser(true);
                job.schedule();
            }
        };
    }

    private static Resource createComposite(Resource compositeType, String defaultName, Resource target) throws DatabaseException {
        Session session = Simantics.getSession();
        session.markUndoPoint();

        Function3<Resource, String, Resource, Resource> instantiator = session.syncRequest( possibleCompositeInstantiator(compositeType) );
        if (instantiator != null)
            return instantiator.apply(compositeType, defaultName, target);

        Resource composite = session.syncRequest( createCompositeRequest( target, defaultName, compositeType ) );
        DefaultActions.asyncPerformDefaultAction(session, composite, false, false, true);
        return composite;
    }

    private static Read<Function3<Resource, String, Resource, Resource>> possibleCompositeInstantiator(final Resource compositeType) {
        return new UniqueRead<Function3<Resource, String, Resource, Resource>>() {
            @Override
            public Function3<Resource, String, Resource, Resource> perform(ReadGraph graph) throws DatabaseException {
                return graph.getPossibleRelatedValue2(
                        compositeType,
                        StructuralResource2.getInstance(graph).Composite_instantiationFunction);
            }
        };
    }

    public static WriteResult<Resource> createCompositeRequest(final Resource target, final String defaultName, final Resource compositeType) {
        return new WriteResultRequest<Resource>() {
            @Override
            public Resource perform(WriteGraph graph) throws DatabaseException {
                return createComposite(graph, target, defaultName, compositeType);
            }
        };
    }

    public static Resource createComposite(WriteGraph graph, Resource target, String defaultName, Resource compositeType) throws DatabaseException {
        String name = NameUtils.findFreshName(graph, defaultName, target, Layer0.getInstance(graph).ConsistsOf, "%s%d");
        return StructuralUtils.newComponent(graph, target, name, compositeType);
    }

}
