package org.simantics.modeling;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.simantics.db.common.utils.VersionInfo;
import org.simantics.utils.strings.AlphanumComparator;

public class CreateVersionDialog extends Dialog {

	private final VersionInfo info;

	private Label errors;
	private Text versionTxT;
	private String result;
	
	public CreateVersionDialog(Shell parentShell, VersionInfo info) {
		super(parentShell);
		this.info = info;
	}
	
	public String mutate(String ver) {
        try {
            int currentInt = Integer.parseInt(ver);
            return "" + (currentInt+1);
        } catch (NumberFormatException e) {
        	if(ver.length() == 1) {
        		char c = ver.charAt(0);
        		c++;
        		Character ch = c; 
        		if(Character.isLetter(ch) && Character.isUpperCase(ch)) return ch.toString();
        	}
        }
        return ver + "_1";
	}
	
	public String getSuggestion() {
		
		String ver = info.version;
		while(validateVersion(ver) != null) ver = mutate(ver);
		return ver;
		
	}

	@Override
	protected Control createDialogArea(Composite parent) {
		Composite c = (Composite) super.createDialogArea(parent);
		
        GridLayoutFactory.fillDefaults().margins(8, 8).numColumns(2).applyTo(c);
        GridDataFactory gd1 = GridDataFactory.fillDefaults().span(2, 1);

        // 1st line
        Label label = new Label(c, 0);
        label.setText("Current version identifier is: " + info.version);
        gd1.applyTo(label);
        
        // 2nd line
        label = new Label(c, 0);
        label.setText("New identifier:");
        GridDataFactory.fillDefaults().applyTo(label);

        versionTxT = new Text(c, SWT.BORDER);
        versionTxT.setText(getSuggestion());
        
        GridDataFactory.fillDefaults().grab(true, false).applyTo(versionTxT);
        
        versionTxT.addModifyListener(new ModifyListener() {

			@Override
			public void modifyText(ModifyEvent e) {

				String result = validateVersion(versionTxT.getText()); 
				if(result != null) {
					errors.setText(result);
					getButton(IDialogConstants.OK_ID).setEnabled(false);
					Shell shell = errors.getShell();
					Point ns = shell.computeSize(SWT.DEFAULT, SWT.DEFAULT);
					Point cs = shell.getSize();
					shell.setSize(Math.max(cs.x, ns.x), Math.max(cs.y, ns.y));
				} else {
					errors.setText("");
					getButton(IDialogConstants.OK_ID).setEnabled(true);
				}
				
			}
        	
        });
        
        // Errors
        errors= new Label(c, 0);
        errors.setText("");
        errors.setForeground(errors.getDisplay().getSystemColor(SWT.COLOR_RED));
        
        gd1.applyTo(errors);
        
		return c;
	}
	
	private String validateVersion(final String version) {
		
		if(info.containsVersion(version)) return "Version already in use";
		if(AlphanumComparator.COMPARATOR.compare(version, info.version) < 0) return "New version must be lexically greater than current";
		return null;
		
	}
	
	@Override
	protected void okPressed() {
		result = versionTxT.getText();
        super.okPressed();
	}
	
	public String getResult() {
		return result;
	}

	@Override
    protected int getShellStyle() {
        return SWT.RESIZE | SWT.TITLE | SWT.CLOSE | SWT.BORDER;
    }

    @Override
    protected void configureShell(Shell newShell) {
        super.configureShell(newShell);
        newShell.setText("Create new version of " + info.baseName);
    }
}
