package org.simantics.modeling.subscription;

import org.eclipse.core.runtime.preferences.IEclipsePreferences;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.Databoard;
import org.simantics.databoard.accessor.reference.ChildReference;
import org.simantics.databoard.accessor.reference.IndexReference;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.util.URIStringUtils;
import org.simantics.db.GraphHints;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.procedure.adapter.TransientCacheAsyncListener;
import org.simantics.db.common.procedure.adapter.TransientCacheListener;
import org.simantics.db.common.request.PossibleIndexRoot;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.exception.PendingVariableException;
import org.simantics.db.layer0.request.PossibleVariableValue;
import org.simantics.db.layer0.request.external.EclipsePreferenceBooleanPrimitiveRead;
import org.simantics.db.layer0.variable.RVI;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.preferences.SubscriptionPreferences;
import org.simantics.utils.datastructures.Pair;
import org.simantics.utils.format.FormattingUtils;
import org.simantics.utils.format.ValueFormat;

public class SubscriptionItemLabel {

    private static final String NO_VARIABLE_ID = "<no variable id>";

    private static EclipsePreferenceBooleanPrimitiveRead showValuesInLabelsRequest =
            new EclipsePreferenceBooleanPrimitiveRead(
                    SubscriptionPreferences.P_NODE,
                    SubscriptionPreferences.P_SUBSCRIPTION_SHOW_VALUE_IN_MODEL_BROWSER,
                    SubscriptionPreferences.DEFAULT_SHOW_VALUE_IN_MODEL_BROWSER);

    public static boolean showValuesInLabel(ReadGraph graph) throws DatabaseException {
        return graph.syncRequest(showValuesInLabelsRequest, TransientCacheListener.instance());
    }

    public static String resolveLabel(ReadGraph graph, Resource item, boolean synchronous) throws DatabaseException {
        IEclipsePreferences chartPreferenceNode = InstanceScope.INSTANCE.getNode( "org.simantics.charts" );
        String s = chartPreferenceNode.get("chart.valueformat", ValueFormat.Default.name());
        ValueFormat valueFormat = ValueFormat.valueOf( s );
        return resolveLabel(graph, item, valueFormat, synchronous);
    }

    public static String resolveLabel(ReadGraph graph, Resource item, ValueFormat valueFormat, boolean synchronous) throws DatabaseException {
        boolean showValue = showValuesInLabel(graph);
        return resolveLabel(graph, item, valueFormat, synchronous, true, showValue);
    }

    public static String resolveLabel(ReadGraph graph, Resource item, ValueFormat valueFormat, boolean synchronous, boolean tryLabel, boolean showValue) throws DatabaseException {

        Layer0 L0 = Layer0.getInstance(graph);
        ModelingResources MOD = ModelingResources.getInstance(graph);

        String label = tryLabel ? graph.<String>getPossibleRelatedValue(item, L0.HasLabel, Bindings.STRING) : null;
        boolean labelDefined = isNonempty(label);
        if (!showValue && labelDefined)
            return label;

        // Create label from rvi
        Binding rviBinding = graph.getService(Databoard.class).getBindingUnchecked( RVI.class );
        RVI rvi = graph.getPossibleRelatedValue(item, MOD.Subscription_Item_VariableId, rviBinding);
        if (rvi == null)
            return NO_VARIABLE_ID;

        Resource model = graph.syncRequest(new PossibleIndexRoot(item));
        if (model != null) {
            ModelContexts contexts = graph.syncRequest(new ModelContextsRequest(model), TransientCacheAsyncListener.<ModelContexts>instance());
            Variable configurationContext = contexts.getConfigurationContext();
            Variable experimentContext = contexts.getExperimentContext();

            if (configurationContext != null) {
                // Resolve variable and take note of the resolution context
                Variable variable = null;
                Variable resolutionContext = experimentContext;
                String resolutionContextURI = contexts.getExperimentContextURI();
                if (showValue && experimentContext != null)
                    variable = rvi.resolvePossible(graph, experimentContext);
                if (variable == null) {
                    resolutionContext = configurationContext;
                    resolutionContextURI = contexts.getConfigurationContextURI();
                    variable = rvi.resolvePossible(graph, configurationContext);
                    if (variable == null && !showValue && experimentContext != null) {
                        // #19656: Ok, if the variable cannot be resolved through the configuration
                        // context make a final effort to resolve it in the experiment context since
                        // it might be a calculation level or UC internal reference.
                        resolutionContext = experimentContext;
                        resolutionContextURI = contexts.getExperimentContextURI();
                        variable = rvi.resolvePossible(graph, experimentContext);
                    }
                }

                if (variable != null) {
                    if (!labelDefined) {
                        label = relativeURI(graph, variable, resolutionContextURI);
                        if (label == null)
                            label = rvi.toPossibleString(graph, configurationContext);
                        if (label != null) {
                            label = removeVariablePrefixPath(label);
                            label = URIStringUtils.unescape(label);
                        }
                    }

                    if (showValue && resolutionContext == experimentContext) {
                        StringBuilder sb = new StringBuilder(label);

                        Double gain = graph.getPossibleRelatedValue(item, MOD.Subscription_Item_Gain, Bindings.DOUBLE);
                        Double bias = graph.getPossibleRelatedValue(item, MOD.Subscription_Item_Bias, Bindings.DOUBLE);
                        String unit = graph.getPossibleRelatedValue(item, MOD.Subscription_Item_Unit, Bindings.STRING);

                        GraphHints old = graph.setHintValue(ReadGraph.GRAPH_HINT_SYNCHRONOUS, synchronous);
                        try {
                            Object value = graph.syncRequest( new PossibleVariableValue<Object>(variable) );
                            sb = formatValue(value, gain, bias, unit, valueFormat, sb);
                        } catch (PendingVariableException e) {
                            sb.append(" (<pending>)");
                        } finally {
                            graph.setHints(old);
                        }

                        label = sb.toString();
                    }
                } else {
                    // Inform user that the subscription item can't be resolved to a value
                    // which means it is possibly referencing a component#property that no
                    // longer exists. 
                    label = label + " (INVALID)";
                }
            }
        }

        return label != null ? label : NO_VARIABLE_ID;
    }

    private static StringBuilder formatValue(Object value, Double gain, Double bias, String unit, ValueFormat valueFormat, StringBuilder sb) {
        if (value instanceof Number) {
            double d = ((Number) value).doubleValue();
            double ad = d;
            if (gain != null)
                ad *= gain;
            if (bias != null)
                ad += bias;
            sb.append(" (").append(valueFormat.format.format(ad));
        } else {
            sb.append(" (").append(FormattingUtils.engineeringFormat(value));
        }
        if (unit != null && !unit.isEmpty()) {
            sb.append(' ').append(unit);
        }
        return sb.append(')');
    }

    public static String removeVariablePrefixPath(String rvi) {
        // Apros-specific logics:
        //   1. remove path prefix
        //   2. change IndexReferences (i-N) into "(N+1)" Apros-style indexing
        int propIndex = rvi.indexOf('#');
        if (propIndex == -1)
            return rvi;
        int prevSlash = rvi.lastIndexOf('/', propIndex);
        if (prevSlash == -1)
            return rvi;
        Pair<String, Integer> attrKey = attributeKey(rvi, propIndex + 1);
        return rvi.substring(prevSlash + 1, propIndex + 1) + attrKey.first
                + (attrKey.second != null ? "(" + (attrKey.second + 1) + ")" : "");
    }

    private static Pair<String, Integer> attributeKey(String key, int start) {
        int iy = key.lastIndexOf('/');
        boolean isIndexed = iy >= start;
        if (isIndexed) {
            ChildReference child = ChildReference.parsePath(key.substring(iy + 1));
            if (child instanceof IndexReference)
                return Pair.make(key.substring(start, iy), ((IndexReference) child).getIndex());
        }
        return Pair.make(key.substring(start), null);
    }

    private static boolean isNonempty(String label) {
        return label != null && !label.isBlank();
    }

    private static String relativeURI(ReadGraph graph, Variable variable, String contextURI) throws DatabaseException {
        return relativeURI(variable.getURI(graph), contextURI);
    }

    private static String relativeURI(String fullURI, String contextURI) {
        return fullURI.substring(contextURI.length());
    }

}
