/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.subscription;

import java.util.UUID;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.Databoard;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.type.Datatype;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.CommentMetadata;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.RVI;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.commandlog.NewSubscriptionItemCommand;
import org.simantics.utils.commandlog.Commands;

/**
 * Request to create new subscription item
 * 
 * @author toni.kalajainen
 */
public class NewSubscriptionItem extends WriteRequest {
	
	Resource subscription;
	Double interval, deadband, gain, bias;
	String unit, label;
	RVI variableId;
	Datatype datatype;
	public Resource subscriptionItem;
	
	public NewSubscriptionItem(Resource subscription, Double interval,
			Double deadband, Double gain, Double bias, String unit,
			String label, RVI variableId, Datatype datatype) {
		super();
		this.subscription = subscription;
		this.interval = interval;
		this.deadband = deadband;
		this.gain = gain;
		this.bias = bias;
		this.unit = unit;
		this.label = label;
		this.variableId = variableId;
		this.datatype = datatype;
	}

	@Override
	public void perform(WriteGraph graph) throws DatabaseException 
	{
		ModelingResources mr = ModelingResources.getInstance(graph);
		Layer0 l0 = Layer0.getInstance(graph);
        subscriptionItem = graph.newResource();
        graph.claim(subscriptionItem, l0.InstanceOf, null, mr.Subscription_Item);
        graph.claim(subscription, l0.ConsistsOf, subscriptionItem);
        graph.claimLiteral(subscriptionItem, l0.HasName, UUID.randomUUID().toString(), Bindings.STRING);
        graph.claimLiteral(subscriptionItem, l0.HasLabel, label, Bindings.STRING);

        if (datatype != null)
            graph.claimLiteral(subscriptionItem, mr.Subscription_Item_Datatype,
                    l0.DataType, datatype,
                    Bindings.getBindingUnchecked(Datatype.class));
        
      	if (interval!=null) graph.claimLiteral(subscriptionItem, mr.Subscription_Item_SamplingInterval, interval, Bindings.DOUBLE);
       	if (deadband!=null) graph.claimLiteral(subscriptionItem, mr.Subscription_Item_Deadband, deadband, Bindings.DOUBLE);
       	if (unit!=null) graph.claimLiteral(subscriptionItem, mr.Subscription_Item_Unit, unit, Bindings.STRING);
       	if (gain!=null) graph.claimLiteral(subscriptionItem, mr.Subscription_Item_Gain, gain, Bindings.DOUBLE);
       	if (bias!=null) graph.claimLiteral(subscriptionItem, mr.Subscription_Item_Bias, bias, Bindings.DOUBLE);
       	
       	Binding rviBinding = graph.getService(Databoard.class).getBindingUnchecked( RVI.class );       	
        graph.claimLiteral(subscriptionItem, mr.Subscription_Item_VariableId, l0.RVI, variableId, rviBinding);
        
        CommentMetadata cm = graph.getMetadata(CommentMetadata.class);
        graph.addMetadata(cm.add("Added new subscription " + label + " " + subscriptionItem + " "));

        if(Commands.isRecording())
            Commands.record(graph, new NewSubscriptionItemCommand(subscriptionItem, subscription, variableId, interval, gain, bias, deadband, unit, label));

	}

}
