package org.simantics.modeling.scl;

import java.io.StringReader;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Future;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.IJobChangeEvent;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.core.runtime.jobs.JobChangeAdapter;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.common.primitiverequest.RelatedValue;
import org.simantics.db.common.uri.ResourceToPossibleURI;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ValidationException;
import org.simantics.layer0.Layer0;
import org.simantics.scl.compiler.commands.CommandSession;
import org.simantics.scl.osgi.SCLOsgi;
import org.simantics.scl.runtime.SCLContext;
import org.simantics.scl.runtime.reporting.SCLReportingHandler;
import org.simantics.utils.datastructures.Pair;
import org.simantics.utils.ui.workbench.WorkbenchUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Tuukka Lehtonen
 * @since 1.46.0
 */
public class SCLScripts {

	private static final Logger LOGGER = LoggerFactory.getLogger(SCLScripts.class);

	private static final String SCL_SCRIPT_CONSOLE_ID = "org.simantics.scl.ui.scriptConsole";

	/**
	 * @param processor database handle
	 * @param script the script to validate
	 * @return <code>null</code> if ok to run script, otherwise a problem description
	 * @throws DatabaseException
	 */
	public static String canRunScript(RequestProcessor processor, final Resource script) throws DatabaseException {
		return null;
	}

	private static Job createRunScriptJob(String scriptName, String scriptText, CommandSession session, SCLReportingHandler handler) {
		Job job = new Job("Run SCL Script") {
			@Override
			protected IStatus run(IProgressMonitor monitor) {
				runScriptWithProgress(monitor, scriptName, scriptText, session, handler);
				return Status.OK_STATUS;
			}
		};
		job.setUser(true);
		return job;
	}

	public static void runScriptWithProgress(String scriptName, String scriptText, CommandSession session, SCLReportingHandler handler) {
		createRunScriptJob(scriptName, scriptText, session, handler).schedule();
	}

	public static void runScriptWithProgress(IProgressMonitor monitor, String scriptName, String scriptText, CommandSession session, SCLReportingHandler handler) {
		monitor.beginTask(scriptName, IProgressMonitor.UNKNOWN);
		try {
			session.execute(new StringReader(scriptText), handler);
		} finally {
			monitor.done();
		}
	}

	private static Optional<SCLReportingHandler> getCurrentReportingHandler() {
		return Optional.ofNullable((SCLReportingHandler) SCLContext.getCurrent().get(SCLReportingHandler.REPORTING_HANDLER));
	}

	private static Optional<SCLReportingHandler> getScriptOutputViewReportingHandler(boolean createIfNecessary) throws PartInitException {
		IAdaptable adaptable = createIfNecessary
				? WorkbenchUtils.showView(SCL_SCRIPT_CONSOLE_ID, IWorkbenchPage.VIEW_VISIBLE)
				: WorkbenchUtils.findView(SCL_SCRIPT_CONSOLE_ID);
		return Optional.ofNullable(adaptable != null ? adaptable.getAdapter(SCLReportingHandler.class) : null);
	}

	public static Pair<CommandSession, SCLReportingHandler> getOrCreateConsoleCommandSession() {
		return getSCLConsoleCommandSession(true);
	}

	public static Pair<CommandSession, SCLReportingHandler> getSCLConsoleCommandSession(boolean createIfNecessary) {
		// Always primarily use current context reporting handler
		Optional<SCLReportingHandler> currentHandler = getCurrentReportingHandler();
		if (currentHandler.isPresent()) {
			return Pair.make(
					new CommandSession(SCLOsgi.MODULE_REPOSITORY, currentHandler.get()),
					currentHandler.get());
		}

		// Fall back to using SCL Script Output view where available or default logging handlers 
		SCLReportingHandler defaultHandler = SCLReportingHandler.DEFAULT_WITHOUT_ECHO;
		if (PlatformUI.isWorkbenchRunning()) {
			try {
				Optional<SCLReportingHandler> handler = getScriptOutputViewReportingHandler(createIfNecessary);
				if (handler.isPresent())
					return Pair.make(new CommandSession(SCLOsgi.MODULE_REPOSITORY, defaultHandler), handler.get());
			} catch (PartInitException e) {
				LOGGER.error("Failed to open SCL Script Output view. Using new CommandSession, reporting to Logger.", e);
			}
		}
		return Pair.make(new CommandSession(SCLOsgi.MODULE_REPOSITORY, defaultHandler), SCLReportingHandler.DEFAULT);
	}

	/**
	 * Schedules a run of the specified SCL script in the background.
	 * 
	 * SCL API:
	 * <pre>
	 * Simantics/SCL/executeSCLScript
	 * </pre>
	 *  
	 * @param resource
	 * @throws DatabaseException
	 */
	public static Future<String> runScript(Resource resource) throws DatabaseException {
		Object graph = SCLContext.getCurrent().get("graph");
		if (graph instanceof ReadGraph)
			throw new DatabaseException("Cannot invoke executeSCLScript from within a database transaction");

		Session s = Simantics.getSession();
		Layer0 L0 = Layer0.getInstance(s);
		String error = canRunScript(s, resource);
		if (error == null) {
			String scriptName = s.syncRequest(new RelatedValue<String>(resource, L0.HasName, Bindings.STRING));
			String scriptText = s.syncRequest(new RelatedValue<String>(resource, L0.SCLScript_definition, Bindings.STRING));
			String scriptUrl = s.syncRequest(new ResourceToPossibleURI(resource));
			Pair<CommandSession, SCLReportingHandler> p = getOrCreateConsoleCommandSession();
			p.first.setRelativeResolutionModuleName(scriptUrl);
			CompletableFuture<String> future = new CompletableFuture<>();
			Job job = createRunScriptJob(scriptName, scriptText, p.first, p.second);
			job.addJobChangeListener(new JobChangeAdapter() {
				@Override
				public void aboutToRun(IJobChangeEvent event) {
					if (future.isCancelled()) {
						event.getJob().cancel();
					}
				}
				@Override
				public void done(IJobChangeEvent event) {
					IStatus r = event.getResult();
					switch (r.getSeverity()) {
					case IStatus.CANCEL:
						future.cancel(true);
						return;
					case IStatus.ERROR:
						Throwable eex = r.getException();
						if (eex != null) {
							future.completeExceptionally(eex);
						} else {
							future.complete(String.format("Script '%s' execution failed with error: %s", scriptName, r.getMessage()));
						}
						return;
					case IStatus.WARNING:
						Throwable wex = r.getException();
						if (wex != null) {
							future.completeExceptionally(wex);
						} else {
							future.complete(String.format("Script '%s' execution completed with warning: %s", scriptName, r.getMessage()));
						}
						return;
					case IStatus.INFO:
					case IStatus.OK:
					default:
						future.complete(String.format("Script '%s' executed", scriptName));
						return;
					}
				}
			});
			job.schedule();
			return future;
		} else {
			throw new ValidationException("Invalid SCL script: " + error);
		}
	}

}
