package org.simantics.modeling.scl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Objects;

import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.UnaryRead;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.SyncListener;
import org.simantics.db.request.Read;
import org.simantics.db.request.ReadExt;
import org.simantics.db.request.RequestFlags;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingUtils;
import org.simantics.modeling.internal.Activator;
import org.simantics.modeling.scl.ontologymodule.OntologyModuleSourceRepository;
import org.simantics.scl.compiler.internal.codegen.types.JavaReferenceValidatorFactory;
import org.simantics.scl.compiler.module.repository.UpdateListener;
import org.simantics.scl.compiler.module.repository.UpdateListener.Observable;
import org.simantics.scl.compiler.source.ModuleSource;
import org.simantics.scl.compiler.source.StringModuleSource;
import org.simantics.scl.compiler.source.repository.ModuleSourceRepository;
import org.simantics.scl.osgi.internal.OsgiJavaReferenceValidatorFactory;
import org.simantics.scl.runtime.SCLContext;
import org.simantics.scl.runtime.tuple.Tuple0;
import org.simantics.structural2.utils.StructuralUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gnu.trove.procedure.TObjectProcedure;
import gnu.trove.set.hash.THashSet;

public enum GraphModuleSourceRepository implements ModuleSourceRepository {
    INSTANCE; 
    
    private static final Logger LOGGER = LoggerFactory.getLogger(OntologyModuleSourceRepository.class);

    private static final OsgiJavaReferenceValidatorFactory REFERENCE_VALIDATOR_FACTORY = new OsgiJavaReferenceValidatorFactory(Activator.getContext().getBundle());

    @Override
    public ModuleSource getModuleSource(final String moduleName, UpdateListener listener) {
        if(!moduleName.startsWith("http://"))
            return null; // Don't do a graph request if this cannot be a resource

        Read<ModuleSource> request = new ReadModuleSource(moduleName);

        try {
            if(listener != null)
                return Simantics.getAvailableRequestProcessor().syncRequest(request, new ModuleListener(listener, moduleName));
            else
                return Simantics.getAvailableRequestProcessor().syncRequest(request);
        } catch (DatabaseException e) {
            LOGGER.error("Failed to read graph module {}.",  moduleName, e);
            return null;
        }
    }

    static class ModuleListener implements SyncListener<ModuleSource>, Observable {
        UpdateListener listener;
        boolean alreadyExecutedOnce;
        final String moduleName;
        public ModuleListener(UpdateListener listener, String moduleName) {
            this.listener = listener;
            this.listener.addObservable(this);
            this.moduleName = moduleName;
        }
        @Override
        public void removeListener(UpdateListener listener) {
            this.listener = null;
        }
        @Override
        public boolean isDisposed() {
            return listener == null;
        }
        private void fireUpdate(ReadGraph graph) {
            if(listener != null) {
                SCLContext context = SCLContext.getCurrent();
                Object oldGraph = context.put("graph", graph);
                try {
                    listener.notifyAboutUpdate();
                } finally {
                    listener = null;
                    context.put("graph", oldGraph);
                }
            }
        }
        @Override
        public void execute(ReadGraph graph, ModuleSource result)
                throws DatabaseException {
            if(alreadyExecutedOnce)
                fireUpdate(graph);
            else
                alreadyExecutedOnce = true;
        }
        @Override
        public void exception(ReadGraph graph, Throwable t)
                throws DatabaseException {
            LOGGER.error("Could not listen {}", this, t);
            if(alreadyExecutedOnce && listener != null)
                fireUpdate(graph);
        }

        @Override
        public String toString() {
            return moduleName + " " + listener + " (" + alreadyExecutedOnce + ") [" + getClass().toString() + "]";
        }
    };
    
    public static class GraphModuleSource extends StringModuleSource {

        protected final boolean immutable;

        public GraphModuleSource(String moduleName, ClassLoader classLoader, String moduleText, boolean immutable) {
            super(moduleName, classLoader, moduleText);
            this.immutable = immutable;
        }
        
        @Override
        public boolean isUpdateable() {
            return !immutable;
        }
        
        @Override
        public void update(String newSourceText) {
            try {
                Simantics.getSession().syncRequest(new WriteModuleSource(getModuleName(), newSourceText));
            } catch (DatabaseException e) {
                LOGGER.error("Could not update {} with newSourceText {}", this, newSourceText);
            }
        }
        
        @Override
        public JavaReferenceValidatorFactory getJavaReferenceValidatorFactory() {
            return REFERENCE_VALIDATOR_FACTORY;
        }

        @Override
        public int hashCode() {
            return Objects.hash(classLoader, immutable, moduleName, moduleText, priority);
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (obj == null)
                return false;
            if (getClass() != obj.getClass())
                return false;
            GraphModuleSource other = (GraphModuleSource) obj;
            return Objects.equals(classLoader, other.classLoader) && immutable == other.immutable
                    && Objects.equals(moduleName, other.moduleName) && Objects.equals(moduleText, other.moduleText)
                    && Double.doubleToLongBits(priority) == Double.doubleToLongBits(other.priority);
        }

    }

    static class PossibleResourceIU extends UnaryRead<String,Resource> implements ReadExt {

        public PossibleResourceIU(String parameter) {
            super(parameter);
        }

        @Override
        public Resource perform(ReadGraph graph) throws DatabaseException {
            return graph.getPossibleResource(parameter);
        }

        @Override
        public boolean isImmutable(ReadGraph graph) throws DatabaseException {
            return false;
        }

        @Override
        public int getType() {
            return RequestFlags.IMMEDIATE_UPDATE;
        }
        
    }
    
    static class ReadModuleSource extends UnaryRead<String, ModuleSource> {
        public ReadModuleSource(String moduleName) {
            super(moduleName);
        }

        @Override
        public ModuleSource perform(ReadGraph graph) throws DatabaseException {
            Resource moduleResource = graph.syncRequest(new PossibleResourceIU(parameter));
            if(moduleResource == null)
                return null;
            Layer0 L0 = Layer0.getInstance(graph);
            if(!graph.isInstanceOf(moduleResource, L0.SCLModule))
                return null;
            String text = graph.getRelatedValue(moduleResource, L0.SCLModule_definition);
            boolean immutable = StructuralUtils.isImmutable(graph, moduleResource);
            return new GraphModuleSource(parameter, getClass().getClassLoader(), text, immutable);
        }
    }
    
    static class WriteModuleSource extends WriteRequest {
        private final String moduleURI;
        private final String sourceText;
        
        public WriteModuleSource(String moduleURI, String sourceText) {
            this.moduleURI = moduleURI;
            this.sourceText = sourceText;
        }

        @Override
        public void perform(WriteGraph graph) throws DatabaseException {
            Resource moduleResource = graph.getPossibleResource(moduleURI);
            if(moduleResource == null)
                return;
            Layer0 L0 = Layer0.getInstance(graph);
            if(!graph.isInstanceOf(moduleResource, L0.SCLModule))
                return;
            graph.claimLiteral(moduleResource, L0.SCLModule_definition, sourceText);
        }
    }

    private THashSet<String> getAllModules(ReadGraph graph) throws DatabaseException {
        THashSet<String> result = new THashSet<String>(); 
        Resource projectResource = Simantics.getProjectResource();
        Layer0 L0 = Layer0.getInstance(graph);
        for(Resource model : graph.getObjects(projectResource, L0.ConsistsOf)) {
            if(graph.isInstanceOf(model, L0.IndexRoot)) {
                for(Resource module : ModelingUtils.searchByType(graph, model, L0.SCLModule))
                    result.add(graph.getURI(module));
            }
        }
        
        Collection<Resource> ontologies = Simantics.applySCL("Simantics/SharedOntologies", "getSharedOntologies", graph, Tuple0.INSTANCE);
        for (Resource ontology : ontologies) {
            for(Resource module : ModelingUtils.searchByType(graph, ontology, L0.SCLModule))
                try {
                    result.add(graph.getURI(module));
                } catch(DatabaseException e) {
                    LOGGER.error("Failed to find uri for " + module + ".");
                }
        }
        
        return result;
    }
    
    @Override
    public void forAllModules(TObjectProcedure<String> procedure) {
        THashSet<String> moduleURIs = null;
        try {
            moduleURIs = Simantics.getAvailableRequestProcessor().syncRequest(new Read<THashSet<String>>() {
                @Override
                public THashSet<String> perform(ReadGraph graph)
                        throws DatabaseException {
                    return getAllModules(graph);
                }
            });
            moduleURIs.forEach(procedure);
        } catch (DatabaseException e) {
            LOGGER.error("Could not execute procedure {} for all modules {}", procedure, String.valueOf(moduleURIs), e);
        }
    }
    
    @Override
    public Collection<String> getModuleNames() {
        ArrayList<String> result = new ArrayList<>();
        forAllModules((String name) -> {
            result.add(name);
            return true;
        });
        return result;
    }
}
