/*******************************************************************************
 * Copyright (c) 2007, 2014 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - #5290
 *******************************************************************************/
package org.simantics.modeling.rules;

import java.util.Collection;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.modeling.ModelingResources;
import org.simantics.structural2.modelingRules.CPConnection;
import org.simantics.structural2.modelingRules.CPConnectionJoin;
import org.simantics.structural2.modelingRules.CPTerminal;
import org.simantics.structural2.modelingRules.IConnectionPoint;

public class Mapping implements IMapping {

	Resource componentMap;
	Resource connectionMap;
	Resource mapsToConnection;
	Resource invConnectionMap;
	Resource componentTypeMap;
	Resource connectionRelationMap;
		
	public Mapping(ReadGraph g) throws DatabaseException {
		initializeRelations(g);
	}

	protected void initializeRelations(ReadGraph g) throws DatabaseException {
		componentMap = g.getResource(ModelingResources.URIs.ElementToComponent);
		connectionMap = g.getResource(ModelingResources.URIs.DiagramConnectionToConnection);
		mapsToConnection = g.getResource(ModelingResources.URIs.MapsToConnection);
		invConnectionMap = g.getResource(ModelingResources.URIs.ConnectionToDiagramConnection);
		componentTypeMap = g.getResource(ModelingResources.URIs.SymbolToComponentType);		
		connectionRelationMap = g.getResource(ModelingResources.URIs.DiagramConnectionRelationToConnectionRelation);
	}

	@Override
	public Resource mapComponentType(ReadGraph g, Resource componentType)
			throws DatabaseException {
		return g.getPossibleObject(componentType, componentTypeMap);
	}

	public Collection<Resource> mapToConnections(ReadGraph g, Resource connection)
			throws DatabaseException {
		return g.getObjects(connection, mapsToConnection);
	}

	@Override
	public Resource mapConnection(ReadGraph g, Resource connection)
			throws DatabaseException {
		return g.getPossibleObject(connection, connectionMap);
	}
	
	@Override
	public Resource invMapConnection(WriteGraph g, Resource connection) 
			throws DatabaseException {
		return g.getPossibleObject(connection, invConnectionMap);
	}
	
	@Override
	public Resource mapConnectionRelation(ReadGraph g, Resource relation)
			throws DatabaseException {
		return g.getPossibleObject(relation, connectionRelationMap);
	}

	@Override
	public Resource mapComponent(ReadGraph g, Resource component)
		throws DatabaseException {
		return g.getPossibleObject(component, componentMap);
	}
	
	@Override
	public IConnectionPoint mapConnectionPoint(ReadGraph g,
			IConnectionPoint cp) throws DatabaseException {
		if(cp instanceof CPConnection) {
			CPConnection ccp = (CPConnection)cp;
			Resource connection = mapConnection(g, ccp.connection);
			if (connection == null)
				return null;
			return new CPConnection(connection);
		}
		if(cp instanceof CPTerminal)
			return mapCPTerminal(g, (CPTerminal)cp);
		if(cp instanceof CPConnectionJoin)
			return cp;
		return null;
	}	

	@Override
	public int mapToConnectionPoints(ReadGraph g, IConnectionPoint cp,
			Collection<IConnectionPoint> result) throws DatabaseException {
		if(cp instanceof CPConnection) {
			CPConnection ccp = (CPConnection)cp;
			Collection<Resource> connections = mapToConnections(g, ccp.connection);
			int size = connections.size();
			if (size > 0)
				for (Resource connection : connections)
					result.add(new CPConnection(connection));
			return size;
		}
		if(cp instanceof CPTerminal) {
			IConnectionPoint mcp = mapCPTerminal(g, (CPTerminal) cp);
			if (mcp != null)
				result.add(mcp);
			return mcp != null ? 1 : 0;
		}
		if(cp instanceof CPConnectionJoin) {
			result.add(cp);
			return 1;
		}
		return 0;
	}

	protected IConnectionPoint mapCPTerminal(ReadGraph g, CPTerminal ccp) throws DatabaseException {
	    if(ccp.component == null)
	        return null;
		Resource component = mapComponent(g, ccp.component);
		if(component == null)
			return null;
		Resource connectionRelation = mapConnectionRelation(g, ccp.relation);
		if(connectionRelation == null)
			return null;
		return new CPTerminal(
				component,
				connectionRelation
			);
	}

}
