package org.simantics.modeling.requests;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Deque;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.function.Predicate;

import org.eclipse.jface.viewers.IFilter;
import org.simantics.scl.runtime.function.Function1;

/**
 * @author Tuukka Lehtonen
 */
public class Nodes {

    public static final Predicate<Node> HAS_PRINTABLES_PREDICATE = n -> n.getDiagramResource() != null || n.hasProperty(Node.PROP_DOC_RESOURCES);
    public static final Predicate<Node> DIAGRAM_RESOURCE_PREDICATE = n -> n.getDiagramResource() != null;
    public static final Predicate<Node> DIAGRAM_RESOURCE_AND_RVI_PREDICATE = n -> n.getDiagramResource() != null && n.getRVI() != null;

    public static Collection<Node> breadthFirstFlatten(IFilter filter, Collection<Node> roots) {
        Collection<Node> result = new ArrayList<>();
        List<Node> sortedRoots = new ArrayList<>(roots);
        Collections.sort(sortedRoots);
        Deque<Node> todo = new ArrayDeque<>(sortedRoots);
        while (!todo.isEmpty()) {
            Node n = todo.removeFirst();
            List<Node> sorted = new ArrayList<>(n.getChildren());
            Collections.sort(sorted);
            todo.addAll(sorted);
            if (filter == null || filter.select(n))
                result.add(n);
        }
        return result;
    }

    public static Collection<Node> depthFirstFlatten(IFilter filter, Collection<Node> roots, Comparator<? super Node> comparator) {
        Collection<Node> result = new ArrayList<>();
        List<Node> sortedRoots = new ArrayList<>(roots);
        Collections.sort(sortedRoots, comparator);
        for (Node n : sortedRoots) {
            depthFirstFlattenRec(filter, comparator, n, result);
        }
        return result;
    }

    private static Collection<Node> depthFirstFlattenRec(IFilter filter, Comparator<? super Node> comparator, Node n, Collection<Node> result) {
        if (filter == null || filter.select(n))
            result.add(n);

        Collection<Node> children = n.getChildren();
        if (children.isEmpty())
            return result;

        List<Node> sorted = new ArrayList<>(children);
        Collections.sort(sorted, comparator);
        for (Node child : sorted)
            depthFirstFlattenRec(filter, comparator, child, result);

        return result;
    }

    /**
     * @param f
     *            function that takes the walked Node as argument and returns a
     *            boolean to describe whether to continue the walk or cancel the
     *            walk. The returned value cannot be <code>null</code>.
     * @return <code>true</code> if the walk was completed or <code>false</code>
     *         if the walk was cancelled
     */
    public static boolean walkTree(Function1<Node, Boolean> filter, Collection<Node> roots) {
        List<Node> sortedRoots = new ArrayList<>(roots);
        Collections.sort(sortedRoots);
        for (Node n : sortedRoots)
            if (!walkTreeRec(filter, n))
               return false;
        return true;
   }

    private static boolean walkTreeRec(Function1<Node, Boolean> filter, Node n) {
        if (!filter.apply(n))
            return false;

        Collection<Node> children = n.getChildren();
        if (!children.isEmpty()) {
            List<Node> sorted = new ArrayList<>(children);
            Collections.sort(sorted);
            for (Node child : sorted)
                if (!walkTreeRec(filter, child))
                    return false;
        }
        return true;
    }

    public static boolean parentIsInSet(Set<Node> set, Node node) {
        for (Node n = node.getParent(); n != null; n = n.getParent())
            if (set.contains(n))
                return true;
        return false;
    }

    public static Set<Node> depthFirstFilter(Predicate<Node> filter, Collection<Node> nodes) {
        Set<Node> result = new TreeSet<>(Node.CASE_INSENSITIVE_COMPARATOR);
        for (Node n : nodes) {
            Node newNode = depthFirstFilterRec(filter, n, null);
            if (newNode != null)
                result.add(newNode);
        }
        return result;
    }

    public static Node depthFirstFilter(Predicate<Node> filter, Node n) {
        return depthFirstFilterRec(filter, n, null);
    }

    private static Node depthFirstFilterRec(Predicate<Node> filter, Node n, Node newParent) {
        Collection<Node> children = n.getChildren();
        if (children.isEmpty())
            return filter.test(n) ? n.cloneWithoutChildren(newParent) : null;

        Node newNode = n.cloneWithoutChildren(newParent);
        int childCount = 0;
        for (Node child : children) {
            Node newChild = depthFirstFilterRec(filter, child, newNode);
            if (newChild != null)
                ++childCount;
        }

        return childCount > 0 ? newNode : null;
    }

}
