package org.simantics.modeling.migration;

import org.eclipse.core.runtime.IProgressMonitor;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.DelayedWriteRequest;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.migration.MigrationState;
import org.simantics.db.layer0.migration.MigrationStep;
import org.simantics.db.layer0.migration.MigrationUtils;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingUtils;
import org.simantics.simulation.ontology.SimulationResource;
import org.simantics.utils.logging.TimeLogger;

/**
 * Runs
 * {@link ModelingUtils#attachCreationInformation(IProgressMonitor, WriteGraph, Resource)}
 * for all imported models.
 * 
 * @author Tuukka Lehtonen
 */
public enum AttachCreationInformationStep implements MigrationStep {

    INSTANCE;

    public static AttachCreationInformationStep get() {
        return INSTANCE;
    }

    @Override
    public void applyTo(final IProgressMonitor monitor, Session session, MigrationState state) throws DatabaseException {
        final Resource indexRoot = MigrationUtils.getResource(monitor, session, state);
        if (indexRoot == null)
            return;

        session.sync(new DelayedWriteRequest() {
            @Override
            public void perform(WriteGraph graph) throws DatabaseException {
                attach(monitor, graph, indexRoot);
            }
        });
    }

    private static void attach(IProgressMonitor monitor, WriteGraph graph, Resource indexRoot) throws DatabaseException {
        Layer0Utils.setDependenciesIndexingDisabled(graph, true);
        Layer0 L0 = Layer0.getInstance(graph);
        SimulationResource SIMU = SimulationResource.getInstance(graph);
        for (Resource model : graph.getObjects(indexRoot, L0.ConsistsOf)) {
            if (graph.isInstanceOf(model, SIMU.Model)) {
                TimeLogger.log(AttachCreationInformationStep.class, "attaching creation information to model " + NameUtils.getSafeName(graph, model));
                ModelingUtils.attachCreationInformation(monitor, graph, model);
            }
        }
    }

}