/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.mapping;

import gnu.trove.map.hash.THashMap;

import java.util.Map;

import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.synchronization.CopyAdvisor;
import org.simantics.diagram.synchronization.ISynchronizationContext;
import org.simantics.diagram.synchronization.graph.GraphCopyAdvisor;
import org.simantics.modeling.ModelingResources;
import org.simantics.utils.datastructures.Pair;

/**
 * @author Tuukka Lehtonen
 */
public class MappedElementCopyAdvisor extends GraphCopyAdvisor {

    protected final CopyAdvisor elementAdvisor;
    protected final CopyAdvisor componentAdvisor;

    public MappedElementCopyAdvisor(CopyAdvisor elementAdvisor, CopyAdvisor componentAdvisor) {
        this.elementAdvisor = elementAdvisor;
        this.componentAdvisor = componentAdvisor;
    }

    @Override
    public Evaluation canCopy(ISynchronizationContext context, WriteGraph graph, Resource source, Resource sourceContainer, Resource targetContainer)
    throws DatabaseException {
        if (CopyAdvisor.SUPPORTED.contains(elementAdvisor.canCopy(context, source, sourceContainer, targetContainer))) {
            ModelingResources MOD = context.get(ModelingSynchronizationHints.MODELING_RESOURCE);
            //Layer0 L0 = Layer0.getInstance(graph);

            Resource component = graph.getPossibleObject(source, MOD.ElementToComponent);
            if (component != null) {
                Resource sourceComposite = sourceContainer != null ? graph.getPossibleObject(sourceContainer, MOD.DiagramToComposite) : null;
                Resource targetComposite = targetContainer != null ? graph.getPossibleObject(targetContainer, MOD.DiagramToComposite) : null;
                //if(targetComposite == null) targetComposite = graph.getPossibleObject(component, L0.PartOf);

                Evaluation componentEval = componentAdvisor.canCopy(context, component, sourceComposite, targetComposite);
                if (componentEval != Evaluation.SUPPORTED)
                    return componentEval;
            }

            Resource connection = graph.getPossibleObject(source, MOD.DiagramConnectionToConnection);
            if (connection != null) {
                Resource sourceComposite = graph.getPossibleObject(sourceContainer, MOD.DiagramToComposite);
                Resource targetComposite = graph.getPossibleObject(targetContainer, MOD.DiagramToComposite);
                //if (targetComposite == null) targetComposite = graph.getPossibleObject(component, L0.PartOf);

                Evaluation connectionEval = componentAdvisor.canCopy(context, connection, sourceComposite, targetComposite);
                if (connectionEval != Evaluation.SUPPORTED)
                    return connectionEval;

                Resource specialConnection = graph.getPossibleObject(source, MOD.DiagramConnectionToConnectionSpecial);
                if (specialConnection != null) {
                    connectionEval = componentAdvisor.canCopy(context, specialConnection, sourceComposite, targetComposite);
                }

                return connectionEval;
            }
            return Evaluation.SUPPORTED;
        }
        return Evaluation.NOT_SUPPORTED;
    }

    @Override
    public Object copy(ISynchronizationContext context, WriteGraph graph, Resource source, Resource sourceContainer, Resource targetContainer) throws DatabaseException {
        return copy(context, graph, source, sourceContainer, targetContainer, new THashMap<Object, Object>());
    }

    @Override
    public Object copy(ISynchronizationContext context, WriteGraph graph, Resource source, Resource sourceContainer, Resource targetContainer, Map<Object, Object> map) throws DatabaseException {
        ModelingResources MOD = context.get(ModelingSynchronizationHints.MODELING_RESOURCE);
        if (MOD == null)
            throw new IllegalStateException("no ModelingSynchronizationHints.MODELING_RESOURCE hint");

        Resource sourceComposite = sourceContainer != null ? graph.getPossibleObject(sourceContainer, MOD.DiagramToComposite) : null;
        Resource targetComposite = targetContainer != null ? graph.getPossibleObject(targetContainer, MOD.DiagramToComposite) : null;
        //if(targetComposite == null) targetComposite = graph.getPossibleObject(component, L0.PartOf);

        Resource elementCopy = (Resource) elementAdvisor.copy(context, source, sourceContainer, targetContainer, map);

        Resource component = graph.getPossibleObject(source, MOD.ElementToComponent);
        Resource componentCopy = null;
        if (component != null) {
            componentCopy = (Resource) componentAdvisor.copy(context, component, sourceComposite, targetComposite, map);
            if (componentCopy != null) {
                graph.claim(elementCopy, MOD.ElementToComponent, componentCopy);
            }
        }

        Resource connection = graph.getPossibleObject(source, MOD.DiagramConnectionToConnection);
        if (connection != null) {
            Resource connectionCopy = (Resource) componentAdvisor.copy(context, connection, sourceComposite, targetComposite, map);
            if (connectionCopy != null) {
                graph.claim(elementCopy, MOD.DiagramConnectionToConnection, connectionCopy);
            }

            Resource specialConnection = graph.getPossibleObject(source, MOD.DiagramConnectionToConnectionSpecial);
            if (specialConnection != null) {
                Resource specialConnectionCopy = (Resource) componentAdvisor.copy(context, specialConnection, sourceComposite, targetComposite, map);
                if (specialConnectionCopy != null) {
                    graph.claim(elementCopy, MOD.DiagramConnectionToConnectionSpecial, specialConnectionCopy);
                }
            }

            // NOTICE: this does not copy MOD.ConnectorToComponent relations.
            // Those are unfortunately reconstructed in Paster since they would
            // be rather difficult to deterministically reconstruct here.
        }

        return elementCopy;
    }

    @Override
    public Object cut(ISynchronizationContext context, WriteGraph graph, Resource source, Resource sourceContainer,
            Resource targetContainer) throws DatabaseException {
        ModelingResources MOD = context.get(ModelingSynchronizationHints.MODELING_RESOURCE);

        Resource sourceComposite = graph.getPossibleObject(sourceContainer, MOD.DiagramToComposite);
        Resource targetComposite = graph.getPossibleObject(targetContainer, MOD.DiagramToComposite);

        Resource component = graph.getPossibleObject(source, MOD.ElementToComponent);
        Resource connection = graph.getPossibleObject(source, MOD.DiagramConnectionToConnection);

        Object elementResult = elementAdvisor.cut(context, source, sourceContainer, targetContainer);
        Object componentResult = null;

        if (sourceComposite != null && targetComposite != null) {
            if (component != null)
                componentResult = componentAdvisor.cut(context, component, sourceComposite, targetComposite);
            if (connection != null)
                componentResult = componentAdvisor.cut(context, connection, sourceComposite, targetComposite);
        }

        if (elementResult == null && componentResult == null)
            return null;
        return Pair.make(elementResult, componentResult);
    }
    
    @Override
    public void onFinish(ISynchronizationContext context) {
    	super.onFinish(context);
    	elementAdvisor.onFinish(context);
    	componentAdvisor.onFinish(context);
    }
    

}
