/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.actions;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.simantics.db.common.NamedResource;
import org.simantics.ui.internal.Activator;
import org.simantics.utils.strings.AlphanumComparator;


/**
 * @author Tuukka Lehtonen
 */
public class NavigationTargetChooserDialog extends Dialog {

    private static final String DIALOG = "NavigationTargetChooserDialog"; //$NON-NLS-1$

    private final NamedResource[] options;

    private final String          title;

    private final String          description;

    private NamedResource         selected;

    private TableViewer           viewer;

    private IDialogSettings       dialogBoundsSettings;

    public NavigationTargetChooserDialog(Shell parent, NamedResource[] options, String title, String description) {
        super(parent);
        this.options = options;
        this.title = title;
        this.description = description;

        IDialogSettings settings = Activator.getDefault().getDialogSettings();
        dialogBoundsSettings = settings.getSection(DIALOG);
        if (dialogBoundsSettings == null)
            dialogBoundsSettings = settings.addNewSection(DIALOG);
    }

    @Override
    protected IDialogSettings getDialogBoundsSettings() {
        return dialogBoundsSettings;
    }

    @Override
    protected void configureShell(Shell newShell) {
        if (title != null) {
            newShell.setText(title);
        } else {
            newShell.setText("Choose Navigation Target");
        }
        super.configureShell(newShell);
    }

    @Override
    protected int getShellStyle() {
        return super.getShellStyle() | SWT.RESIZE;
    }

    @Override
    protected Point getInitialSize() {
        Point defaultSize = getShell().computeSize(SWT.DEFAULT, SWT.DEFAULT, true);
        Point result = super.getInitialSize();
        if (defaultSize.equals(result))
            return new Point(400, 500);
        return result;
    }

    @Override
    protected Control createDialogArea(Composite parent) {
        Composite composite = (Composite) super.createDialogArea(parent);

        if (description != null) {
            Label label = new Label(composite, 0);
            label.setText(description);
            label.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));
        }

        viewer = new TableViewer(composite, SWT.V_SCROLL | SWT.H_SCROLL | SWT.BORDER | SWT.SINGLE);
        viewer.getControl().setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
        viewer.setContentProvider(new ArrayContentProvider());
        viewer.setLabelProvider(new TargetLabelProvider());
        viewer.setSorter(sorter);
        viewer.setInput(options);

        viewer.addDoubleClickListener(new IDoubleClickListener() {
            @Override
            public void doubleClick(DoubleClickEvent event) {
                okPressed();
            }
        });

        viewer.addSelectionChangedListener(new ISelectionChangedListener() {
            @Override
            public void selectionChanged(SelectionChangedEvent event) {
                NavigationTargetChooserDialog.this.selectionChanged(event.getSelection());
            }
        });

        if (options.length > 0) {
            viewer.setSelection(new StructuredSelection(options[0]), true);
        }

        applyDialogFont(composite);
        return composite;
    }

    private void selectionChanged(ISelection s) {
        Button ok = getButton(IDialogConstants.OK_ID);
        IStructuredSelection iss = (IStructuredSelection) s;
        if (iss == null || iss.isEmpty()) {
            if (ok != null)
                ok.setEnabled(false);
            return;
        }

        if (ok != null)
            ok.setEnabled(true);
        return;
    }

    @Override
    protected void okPressed() {
        selected = (NamedResource) ((IStructuredSelection) viewer.getSelection()).getFirstElement();
        super.okPressed();
    }

    public NamedResource getSelection() {
        return selected;
    }

    private final ViewerSorter sorter = new ViewerSorter() {
        @Override
        public int category(Object element) {
            return 0;
        }
        @Override
        public int compare(Viewer viewer, Object e1, Object e2) {
            NamedResource a1 = (NamedResource) e1;
            NamedResource a2 = (NamedResource) e2;
            return AlphanumComparator.CASE_INSENSITIVE_COMPARATOR.compare(a1.getName(), a2.getName());
        }
    };

    class TargetLabelProvider extends LabelProvider {
        @Override
        public Image getImage(Object element) {
            return null;
        }

        @Override
        public String getText(Object element) {
            NamedResource a = (NamedResource) element;
            return a.getName();
        }
    }

}
