/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.ResourceArray;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.layer0.Layer0;
import org.simantics.simulation.ontology.SimulationResource;
import org.simantics.structural.stubs.StructuralResource2;

/**
 * @author Tuukka Lehtonen
 */
public final class ComponentUtils {

    public static boolean componentHasDiagram(ReadGraph g, Resource component) throws DatabaseException {
        return !getComponentDiagrams(g, component).isEmpty();
    }

    public static boolean compositeHasDiagram(ReadGraph g, Resource composite) throws DatabaseException {
        return !getCompositeDiagrams(g, composite).isEmpty();
    }

    public static Collection<Resource> getComponentDiagrams(ReadGraph g, Resource component) throws DatabaseException {
        StructuralResource2 sr = StructuralResource2.getInstance(g);
        Resource componentType = g.getPossibleType(component, sr.Component);
        Resource composite = componentType != null ? g.getPossibleObject(componentType, sr.IsDefinedBy) : null;
        if (composite != null)
            return getCompositeDiagrams(g, composite);
        return Collections.emptyList();
    }

    public static Collection<Resource> getCompositeDiagrams(ReadGraph g, Resource composite) throws DatabaseException {
        return g.getObjects(composite, ModelingResources.getInstance(g).CompositeToDiagram);
    }

    public static Resource getCompositeDiagram(ReadGraph g, Resource composite) throws DatabaseException {
        return g.getSingleObject(composite, ModelingResources.getInstance(g).CompositeToDiagram);
    }

    public static Resource getPossibleCompositeDiagram(ReadGraph g, Resource composite) throws DatabaseException {
        return g.getPossibleObject(composite, ModelingResources.getInstance(g).CompositeToDiagram);
    }

    public static Resource getPossibleDiagramComposite(ReadGraph g, Resource diagram) throws DatabaseException {
        return g.getPossibleObject(diagram, ModelingResources.getInstance(g).DiagramToComposite);
    }

    public static Collection<ResourceArray> formInputs(ReadGraph g, Resource compositeOrComponent) throws DatabaseException {
        ResourceArray path = getStructuralPath(g, compositeOrComponent);
        Collection<ResourceArray> inputs = new ArrayList<ResourceArray>();

        if (compositeHasDiagram(g, compositeOrComponent)) {
            for (Resource dia : getCompositeDiagrams(g, compositeOrComponent)) {
                //System.out.println(": " + NameUtils.getSafeName(g, compositeOrComponent));
                //System.out.println(": " + NameUtils.getSafeName(g, dia));
                inputs.add(path.prepended(dia));
            }
        } else if (componentHasDiagram(g, compositeOrComponent)) {
            for (Resource dia : getComponentDiagrams(g, compositeOrComponent)) {
                //System.out.println("* " + NameUtils.getSafeName(g, compositeOrComponent));
                //System.out.println("* " + NameUtils.getSafeName(g, dia));
                inputs.add(path.prepended(dia));
            }
        }

        return inputs;
    }

//    public static ResourceArray getStructuralPath(ReadGraph g, Resource compositeOrComponent) throws DatabaseException {
//        Builtins b = g.getBuiltins();
//        StructuralResource2 sr = StructuralResource2.getInstance(g);
//        Resource component = null;
//
//        List<Resource> result = new ArrayList<Resource>();
//
//        // Find component where to start the path browsing
//        if (g.isInstanceOf(compositeOrComponent, sr.Composite)) {
////            System.out.println("got composite: " + NameUtils.getSafeName(g, compositeOrComponent));
//            Resource defines = g.getPossibleObject(compositeOrComponent, sr.Defines);
//            if (defines == null)
//                return ResourceArray.EMPTY;
//            component = g.getPossibleObject(defines, b.HasSingleInstance);
//            if (component == null)
//                return ResourceArray.EMPTY;
////            System.out.println("start path browsing from: " + NameUtils.getSafeName(g, component));
//        } else if (g.isInstanceOf(compositeOrComponent, sr.Component)) {
////            System.out.println("got component: " + NameUtils.getSafeName(g, compositeOrComponent));
//            component = compositeOrComponent;
//        } else {
//            return ResourceArray.EMPTY;
//        }
//
//        while (component != null) {
////            System.out.println("ADDING COMPONENT: " + NameUtils.getSafeName(g, component));
//            result.add(component);
//
//            Resource composite = g.getPossibleObject(component, b.PartOf);
//            if (composite == null)
//                break;
////            System.out.println("  PART OF: " + NameUtils.getSafeName(g, composite));
//
//            Resource defines = g.getPossibleObject(composite, sr.Defines);
//            if (defines == null)
//                break;
////            System.out.println("  WHICH DEFINES: " + NameUtils.getSafeName(g, defines));
//
//            component = g.getPossibleObject(defines, b.HasSingleInstance);
//        }
////        System.out.println("FOUND PATH:");
////        for (Resource r : result)
////            System.out.println("    " + NameUtils.getSafeName(g, r));
//        return new ResourceArray(result);
//    }

    public static ResourceArray getStructuralPath(ReadGraph g, Resource compositeOrComponent) throws DatabaseException {
        Layer0 b = Layer0.getInstance(g);
        StructuralResource2 sr = StructuralResource2.getInstance(g);
        Resource component = null;

        List<Resource> result = new ArrayList<Resource>();

//        System.out.println("checking composite or component: " + NameUtils.getSafeName(g, compositeOrComponent));

        // Find component where to start the path browsing
        if (g.isInstanceOf(compositeOrComponent, sr.Composite)) {
//            System.out.println("got composite: " + NameUtils.getSafeName(g, compositeOrComponent));
            component = compositeOrComponent;
        } else if (g.isInstanceOf(compositeOrComponent, sr.Component)) {
//            System.out.println("got component: " + NameUtils.getSafeName(g, compositeOrComponent));
            component = compositeOrComponent;
        } else {
            return ResourceArray.EMPTY;
        }
//        System.out.println("start path browsing from: " + NameUtils.getSafeName(g, component));

        while (component != null) {
//            System.out.println("ADDING COMPONENT: " + NameUtils.getSafeName(g, component));
            result.add(component);

            Resource composite = g.getPossibleObject(component, b.PartOf);
            if (composite == null)
                break;
            if (!g.isInstanceOf(composite, sr.Composite)) {
//                System.out.println("  PART OF NON-COMPOSITE: " + NameUtils.getSafeName(g, composite));
                break;
            }
//            System.out.println("  PART OF COMPOSITE: " + NameUtils.getSafeName(g, composite));
            component = composite;
        }
//        System.out.println("FOUND PATH:");
//        for (Resource r : result)
//            System.out.println("    " + NameUtils.getSafeName(g, r));
        return new ResourceArray(result);
    }

    public static Resource getCompositeConfigurationRoot(ReadGraph g, Resource composite) throws DatabaseException {
        StructuralResource2 sr = StructuralResource2.getInstance(g);

        if (!g.isInstanceOf(composite, sr.Composite))
            throw new IllegalArgumentException("argument " + NameUtils.getSafeName(g, composite) + " is not a composite component");

        ResourceArray path = getStructuralPath(g, composite);
        if (path.isEmpty()) {
            return composite;
        }

        Resource topmostComponent = path.resources[path.resources.length - 1];
        //Resource parent = g.getPossibleObject(topmostComponent, g.getBuiltins().PartOf);
        //assert parent != null;
        //return parent;
        return topmostComponent;
    }

    public static Resource getComponentConfigurationRoot(ReadGraph g, Resource component) throws DatabaseException {
        StructuralResource2 sr = StructuralResource2.getInstance(g);

        if (g.isInstanceOf(component, sr.Composite))
            throw new IllegalArgumentException("argument " + NameUtils.getSafeName(g, component) + " is a composite component");
        if (!g.isInstanceOf(component, sr.Component))
            throw new IllegalArgumentException("argument " + NameUtils.getSafeName(g, component) + " is not a component");

        ResourceArray path = getStructuralPath(g, component);
        Resource topmostComponent = path.isEmpty() ? component : path.resources[path.resources.length - 1];
//        Resource parent = g.getPossibleObject(topmostComponent, g.getBuiltins().PartOf);
//        assert parent != null;
//        return parent;
        return topmostComponent;
    }

    public static Resource tryGetComponentContainer(ReadGraph g, Resource component) throws DatabaseException {
        Layer0 l0 = Layer0.getInstance(g);
        StructuralResource2 sr = StructuralResource2.getInstance(g);

        if (g.isInstanceOf(component, sr.Composite))
            throw new IllegalArgumentException("argument " + NameUtils.getSafeName(g, component) + " is a composite component");
        if (!g.isInstanceOf(component, sr.Component))
            throw new IllegalArgumentException("argument " + NameUtils.getSafeName(g, component) + " is not a component");

        Resource container = g.getPossibleObject(component, l0.PartOf);
        if (container == null)
            return null;
        return container;
    }

    public static Resource tryGetComponentConfigurationRoot(ReadGraph g, Resource component) throws DatabaseException {
        StructuralResource2 sr = StructuralResource2.getInstance(g);

        if (g.isInstanceOf(component, sr.Composite))
            return null;
        if (!g.isInstanceOf(component, sr.Component))
            return null;

        ResourceArray path = getStructuralPath(g, component);
        Resource topmostComponent = path.isEmpty() ? component : path.resources[path.resources.length - 1];

//        Resource parent = g.getPossibleObject(topmostComponent, g.getBuiltins().PartOf);
//        return parent;
        return topmostComponent;
    }

    public static List<Resource> getCompositePathToConfiguration(ReadGraph graph, Resource composite) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        SimulationResource SIMU = SimulationResource.getInstance(graph);
        LinkedList<Resource> compositePath = new LinkedList<Resource>();
        while (!graph.hasStatement(composite, SIMU.IsConfigurationOf)) {
            compositePath.addFirst(composite);
            composite = graph.getPossibleObject(composite, L0.PartOf);
            if (composite == null)
                return null;
        }
        return compositePath;
    }

}
