/*******************************************************************************
 * Copyright (c) 2007, 2011, 2014 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart.properties.xyline;

import java.util.List;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.ui.IWorkbenchSite;
import org.simantics.Simantics;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.swt.SingleSelectionInputSource;
import org.simantics.browsing.ui.swt.widgets.Button;
import org.simantics.browsing.ui.swt.widgets.GraphExplorerComposite;
import org.simantics.browsing.ui.swt.widgets.impl.SelectionListenerImpl;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupport;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupportImpl;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.PossibleObjectWithType;
import org.simantics.db.common.utils.ListUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.RemoverUtil;
import org.simantics.db.management.ISessionContext;
import org.simantics.db.request.Read;
import org.simantics.jfreechart.chart.ChartUtils;
import org.simantics.jfreechart.chart.properties.AdjustableTab;
import org.simantics.layer0.Layer0;
import org.simantics.sysdyn.JFreeChartResource;
import org.simantics.utils.datastructures.ArrayMap;
import org.simantics.utils.ui.AdaptionUtils;

/**
 * PropertyTab displaying properties of axis and variables of a chart
 *  
 * @author Teemu Lempinen
 * @author Tuomas Miettinen
 *
 */
public class XYLineAxisAndVariablesTab extends AdjustableTab {

    private GraphExplorerComposite explorer;
    private ScrolledComposite propertyContainer;
    private Button addAxis, addVariable, remove;
    private WidgetSupportImpl additionalSupport;
	private Composite buttonComposite;

    public XYLineAxisAndVariablesTab(Object id) {
        super(id);
        additionalSupport = new WidgetSupportImpl();
    }

    /**
     * Updates the content of propertyContainer  
     * @param context
     */
    private void updateSelection(ISessionContext context) {
        ISelectionProvider selectionProvider = (ISelectionProvider)explorer.getAdapter(ISelectionProvider.class);
        IStructuredSelection selection = (IStructuredSelection)selectionProvider.getSelection();
        final Resource resource = AdaptionUtils.adaptToSingle(selection, Resource.class);
        if(resource == null)
            return;

        // Get the type of the selected node (axis or series)
        String typeUri = null;
        try {
            typeUri = Simantics.getSession().syncRequest(new Read<String>() {

                @Override
                public String perform(ReadGraph graph) throws DatabaseException {
                    JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
                    if(graph.isInstanceOf(resource, jfree.Axis))
                        return graph.getURI(jfree.Axis);
                    else if (graph.isInstanceOf(resource, jfree.Series))
                        return graph.getURI(jfree.Series);
                    return null;
                }

            });
        } catch (DatabaseException e) {
            e.printStackTrace();
        }

        // Create a PropertyComposite for the selected node
        if(typeUri != null) {

            for(Control child : propertyContainer.getChildren()) {
                child.dispose();
            }

            if(typeUri.equals(JFreeChartResource.URIs.Axis)) {
                AxisPropertyComposite apc = new AxisPropertyComposite(propertyContainer, context, additionalSupport, SWT.NONE, isVertical());
                propertyContainer.setContent(apc);
                Point size = apc.computeSize(SWT.DEFAULT, SWT.DEFAULT);
                propertyContainer.setMinSize(size);
            } else if(typeUri.equals(JFreeChartResource.URIs.Series)) {
                SeriesPropertyComposite spc = new SeriesPropertyComposite(propertyContainer, context, additionalSupport, SWT.NONE);
                propertyContainer.setContent(spc);
                Point size = spc.computeSize(SWT.DEFAULT, SWT.DEFAULT);
                propertyContainer.setMinSize(size);
            }
        }

        additionalSupport.fireInput(context, selection);
    }

    /**
     * SelectionListener for adding a new range axis to a plot
     * @author Teemu Lempinen
     *
     */
    private class NewAxisListener extends SelectionListenerImpl<Resource> {

        public NewAxisListener(ISessionContext context) {
            super(context);
        }

        @Override
        public void apply(WriteGraph graph, Resource chart) throws DatabaseException {
            JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
            Layer0 l0 = Layer0.getInstance(graph);
            Resource plot = graph.syncRequest(new PossibleObjectWithType(chart, l0.ConsistsOf, jfree.Plot));
            if(plot != null) {
                Resource rangeAxis = ChartUtils.createNumberRangeAxis(graph, plot);
                if(rangeAxis != null) {
                    Resource domainAxis = graph.getPossibleObject(plot, jfree.Plot_domainAxis);
                    ChartUtils.createXYDataset(graph, plot, domainAxis, rangeAxis);
                }
            }
        }
    }


    /**
     * SelectionListener for adding a new variable to a plot
     * @author Teemu Lempinen
     *
     */
    private class NewVariableListener extends SelectionListenerImpl<Resource> {

        public NewVariableListener(ISessionContext context) {
            super(context);
        }

        @Override
        public void apply(WriteGraph graph, Resource input) throws DatabaseException {
            NodeContext nc = explorer.getExplorer().getRoot();
            if(nc == null)
                return;
            
            JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
            Layer0 l0 = Layer0.getInstance(graph);
            
            if(input == null) {
                Resource chart = AdaptionUtils.adaptToSingle(nc, Resource.class);
                if(chart == null) return;
                Resource plot = graph.syncRequest(new PossibleObjectWithType(chart, l0.ConsistsOf, jfree.Plot));
                if(plot == null) return;
                Resource rangelist = graph.getPossibleObject(plot, jfree.Plot_rangeAxisList);
                if(rangelist == null) return;
                List<Resource> list = ListUtils.toList(graph, rangelist);
                if(list == null || list.isEmpty()) return;
                input = list.get(0);
            }
            
            Resource dataset;
            if(graph.isInstanceOf(input, jfree.Series)) {
                // Selected resource is series. Add to same dataset
                dataset = graph.getPossibleObject(input, l0.PartOf);
            } else {
                // Selected resource is axis. Find the dataset it is mapped to or create dataset if not created already
                dataset = graph.getPossibleObject(input, jfree.Dataset_mapToRangeAxis_Inverse);
                if(dataset == null) {
                    Resource plot = graph.getPossibleObject(input, l0.PartOf);
                    if(plot == null) return;
                    Resource domainAxis = graph.getPossibleObject(plot, jfree.Plot_domainAxis);
                    if(domainAxis == null) return;
                    ChartUtils.createXYDataset(graph, plot, domainAxis, input);
                }
            }

            if(dataset != null) {
                // Create series with no rvi
                ChartUtils.createSeries(graph, dataset, null);
            }
        }
    }


    /**
     * SelectionListener for remove button
     * @author Teemu Lempinen
     *
     */
    private class RemoveListener extends SelectionListenerImpl<Resource> {

        public RemoveListener(ISessionContext context) {
            super(context);
        }

        /**
         * Removes selected resource from explorer
         */
        @Override
        public void apply(WriteGraph graph, Resource input) throws DatabaseException {
            if(input == null)
                return; 

            JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
            Layer0 l0 = Layer0.getInstance(graph);
            Resource list = null;
            if(graph.isInstanceOf(input, jfree.Series)) {
                // Remove series from dataset and seriesList
                Resource dataset = graph.getPossibleObject(input, l0.PartOf);
                if(dataset != null)
                    list = graph.getPossibleObject(dataset, jfree.Dataset_seriesList);
            } else {
                // Remove associated dataset
                Resource dataset = graph.getPossibleObject(input, jfree.Dataset_mapToRangeAxis_Inverse);
                if(dataset != null) {
                    graph.deny(dataset, jfree.Dataset_mapToDomainAxis);
                    graph.deny(dataset, jfree.Dataset_mapToRangeAxis);
                    RemoverUtil.remove(graph, dataset);
                }

                // Remove axis from plot and rangeAxisList
                Resource plot = graph.getPossibleObject(input, l0.PartOf);
                if(plot != null)
                    list = graph.getPossibleObject(plot, jfree.Plot_rangeAxisList);
            }
            if(list != null)
                ListUtils.removeElement(graph, list, input);
            RemoverUtil.remove(graph, input);
        }
    }


	@Override
	protected void createAndAddControls(Composite body, IWorkbenchSite site,
			final ISessionContext context, WidgetSupport support) {
		composite = new Composite(body, SWT.NONE);

        // (Ontology-based) GraphExplorer displaying range axis and variables mapped to those axis
        explorer = new AxisAndVariablesExplorerComposite(ArrayMap.keys(
                "displaySelectors", "displayFilter").values(false, false), site, composite, support, SWT.FULL_SELECTION | SWT.BORDER | SWT.SINGLE);
        explorer.setBrowseContexts(JFreeChartResource.URIs.ChartAxisAndVariablesBrowseContext);
        explorer.setInputSource(new SingleSelectionInputSource(
                Resource.class));
        explorer.getExplorer().setAutoExpandLevel(2); // Expand everything in the beginning
        explorer.finish();

        ((Tree)explorer.getExplorerControl()).addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                updateSelection(context);
            }
        });

        // Scrolled composite for displaying properties of a selection in explorer
        propertyContainer = new ScrolledComposite(composite, SWT.H_SCROLL | SWT.V_SCROLL);
        propertyContainer.setExpandHorizontal(true);
        propertyContainer.setExpandVertical(true);

        // Buttons for adding axis and variables and removing selected items from explorer
        buttonComposite = new Composite(composite, SWT.NONE);

        addAxis = new Button(buttonComposite, support, SWT.NONE);
        addAxis.setText("Add axis");
        addAxis.addSelectionListener(new NewAxisListener(context));

        addVariable = new Button(buttonComposite, additionalSupport, SWT.NONE);
        addVariable.setText("Add variable");
        addVariable.addSelectionListener(new NewVariableListener(context));

        remove = new Button(buttonComposite, additionalSupport, SWT.NONE);
        remove.setText("Remove");
        remove.addSelectionListener(new RemoveListener(context));
	}

	@Override
	protected void createControlLayoutVertical() {
        GridDataFactory.fillDefaults().grab(true, true).applyTo(composite);
        GridLayoutFactory.fillDefaults().numColumns(1).margins(3, 3).applyTo(composite);

        GridDataFactory.fillDefaults().hint(220, SWT.DEFAULT).grab(false, true).applyTo(explorer);

        // Scrolled composite for displaying properties of a selection in explorer
        GridDataFactory.fillDefaults().hint(SWT.DEFAULT, 210).span(1, 1).grab(true, false).applyTo(propertyContainer);
        GridLayoutFactory.fillDefaults().applyTo(propertyContainer);

        // Buttons for adding axis and variables and removing selected items from explorer
        GridDataFactory.fillDefaults().applyTo(buttonComposite);
        GridLayoutFactory.fillDefaults().numColumns(3).applyTo(buttonComposite);
	}

	@Override
	protected void createControlLayoutHorizontal(boolean wideScreen) {
        GridDataFactory.fillDefaults().grab(true, true).applyTo(composite);
        GridLayoutFactory.fillDefaults().numColumns(2).margins(3, 3).applyTo(composite);

        GridDataFactory.fillDefaults().hint(250, SWT.DEFAULT).grab(false, true).applyTo(explorer);

        // Scrolled composite for displaying properties of a selection in explorer
        GridDataFactory.fillDefaults().hint(SWT.DEFAULT, SWT.DEFAULT).span(1, 2).grab(true, true).applyTo(propertyContainer);
        GridLayoutFactory.fillDefaults().applyTo(propertyContainer);

        // Buttons for adding axis and variables and removing selected items from explorer
        GridDataFactory.fillDefaults().applyTo(buttonComposite);
        GridLayoutFactory.fillDefaults().numColumns(3).applyTo(buttonComposite);
	}
}
