/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart.properties.xyline;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Spinner;
import org.simantics.browsing.ui.swt.widgets.StringPropertyFactory;
import org.simantics.browsing.ui.swt.widgets.StringPropertyModifier;
import org.simantics.browsing.ui.swt.widgets.TrackedText;
import org.simantics.browsing.ui.swt.widgets.impl.ReadFactoryImpl;
import org.simantics.browsing.ui.swt.widgets.impl.TextModifyListener;
import org.simantics.browsing.ui.swt.widgets.impl.TrackedModifyEvent;
import org.simantics.browsing.ui.swt.widgets.impl.Widget;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupport;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.management.ISessionContext;
import org.simantics.jfreechart.chart.properties.ColorPicker;
import org.simantics.jfreechart.chart.properties.JFreeChartPropertyColorProvider;
import org.simantics.jfreechart.chart.properties.RVIFactory;
import org.simantics.jfreechart.chart.properties.RVIModifier;
import org.simantics.jfreechart.chart.properties.RangeComposite;
import org.simantics.jfreechart.chart.properties.TrackedSpinner;
import org.simantics.jfreechart.chart.properties.VariableExistsValidator;
import org.simantics.layer0.Layer0;
import org.simantics.sysdyn.JFreeChartResource;
import org.simantics.utils.ui.AdaptionUtils;

/**
 * Composite for displaying series properties in {@link XYLineAxisAndVariablesTab}
 * 
 * @author Teemu Lempinen
 *
 */
public class SeriesPropertyComposite extends Composite {
     
    
    public SeriesPropertyComposite(Composite parent, ISessionContext context, WidgetSupport support, int style) {
        super(parent, style);
        createContent(context, support);
    }
    
    protected void createContent(ISessionContext context, WidgetSupport support) {

        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(2).applyTo(this);

        // Variable for the series
        variable(this, context, support);

        // Range
        range(this, context, support);
        
        // Label to be displayed in chart for this series 
        seriesLabel(this, context, support);

        // Color
        color(this, context, support);

        // Line width
        lineWidth(this, support);

    }
    
    protected TrackedText variable(Composite container, ISessionContext context, WidgetSupport support) {
        Label label = new Label(container, SWT.NONE);
        label.setText("Variable:");
        GridDataFactory.fillDefaults().align(SWT.END, SWT.FILL).applyTo(label);

        TrackedText variable = new TrackedText(container, support, SWT.BORDER);
        variable.setTextFactory(new RVIFactory());
        variable.addModifyListener(new RVIModifier(variable.getWidget(), support));
        variable.setColorProvider(new JFreeChartPropertyColorProvider(variable.getResourceManager()));
        variable.setInputValidator(new VariableExistsValidator(support, variable));
        GridDataFactory.fillDefaults().grab(true, false).applyTo(variable.getWidget());
        return variable;
    }
    
    protected RangeComposite range(Composite container, ISessionContext context, WidgetSupport support) {
        Label label = new Label(container, SWT.NONE);
        label.setText("Range:");
        GridDataFactory.fillDefaults().align(SWT.END, SWT.CENTER).applyTo(label);
        
        RangeComposite rangeComposite = new RangeComposite(container, context, support, SWT.NONE);
        GridDataFactory.fillDefaults().grab(true, false).applyTo(rangeComposite);
        return rangeComposite;
    }
    
    protected TrackedText seriesLabel(Composite container, ISessionContext context, WidgetSupport support) {
        Label label = new Label(container, SWT.NONE);
        label.setText("Label:");
        GridDataFactory.fillDefaults().align(SWT.END, SWT.FILL).applyTo(label);

        TrackedText labelText = new TrackedText(container, support, SWT.BORDER);
        labelText.setTextFactory(new StringPropertyFactory(Layer0.URIs.HasLabel, ""));
        labelText.addModifyListener(new StringPropertyModifier(context, Layer0.URIs.HasLabel));
        labelText.setColorProvider(new JFreeChartPropertyColorProvider(labelText.getResourceManager()));
        GridDataFactory.fillDefaults().grab(true, false).applyTo(labelText.getWidget());
        return labelText;
    }
    
    protected ColorPicker color(Composite container, ISessionContext context, WidgetSupport support) {
        Label label = new Label(container, SWT.NONE);
        label.setText("Color:");
        GridDataFactory.fillDefaults().align(SWT.END, SWT.CENTER).applyTo(label);

        ColorPicker colorPicker = new ColorPicker(container, context, support, SWT.NONE);
        GridDataFactory.fillDefaults().grab(true, false).applyTo(colorPicker);
        return colorPicker;
    }
    
    protected TrackedSpinner lineWidth(Composite container, WidgetSupport support) {
        Label label = new Label(container, SWT.NONE);
        label.setText("Line width:");
        GridDataFactory.fillDefaults().align(SWT.END, SWT.FILL).applyTo(label);

        TrackedSpinner width = new TrackedSpinner(container, support, SWT.BORDER);
        width.setSelectionFactory(new WidthSelectionFactory());
        width.addModifyListener(new WidthModifyListener());
        width.setMinimum(1);
        width.setMaximum(10);
        return width;
    }


    /**
     * ModifyListener for the width {@link TrackedSpinner}
     * 
     * @author Teemu Lempinen
     *
     */
    private class WidthModifyListener implements TextModifyListener, Widget {

        private ISessionContext context;
        private Object lastInput = null;

        @Override
        public void modifyText(TrackedModifyEvent e) {
            if(context == null)
                return;

            // Get the text value from spinner and associated resource (input)
            Spinner spinner = (Spinner)e.getWidget();
            final String textValue = spinner.getText();
            final Object input = lastInput;

            try {
                context.getSession().syncRequest(new WriteRequest() {

                    @Override
                    public void perform(WriteGraph graph) throws DatabaseException {
                        // Apply with (textValue) to the series (input)
                        Resource series = AdaptionUtils.adaptToSingle(input, Resource.class);
                        JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
                        try {
                            // usually reliable, since the spinner does all the checks
                            Integer value = Integer.parseInt(textValue); 
                            graph.claimLiteral(series, jfree.Series_lineWidth, value, Bindings.INTEGER);
                        } catch (NumberFormatException e) {
                            e.printStackTrace();
                        }
                    }

                });
            } catch (DatabaseException e1) {
                e1.printStackTrace();
            }
        }

        @Override
        public void setInput(ISessionContext context, Object parameter) {
            this.context = context;
            lastInput = parameter;
        }

    }

    /**
     * Class for setting the value for width {@link TrackedSpinner}
     * @author Teemu Lempinen
     *
     */
    private class WidthSelectionFactory extends ReadFactoryImpl<Resource, Integer>   {

        @Override
        public Integer perform(ReadGraph graph, Resource axis) throws DatabaseException {
            JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
            Integer width = graph.getPossibleRelatedValue(axis, jfree.Series_lineWidth);
            if(width == null)
                // Default width == 1
                width = 1;
            return width;
        }

    }
}
