/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart.properties.pie;

import java.util.Collection;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.ui.IWorkbenchSite;
import org.simantics.browsing.ui.swt.SingleSelectionInputSource;
import org.simantics.browsing.ui.swt.widgets.Button;
import org.simantics.browsing.ui.swt.widgets.GraphExplorerComposite;
import org.simantics.browsing.ui.swt.widgets.impl.SelectionListenerImpl;
import org.simantics.browsing.ui.swt.widgets.impl.Widget;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupport;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupportImpl;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.PossibleObjectWithType;
import org.simantics.db.common.utils.ListUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.RemoverUtil;
import org.simantics.db.management.ISessionContext;
import org.simantics.jfreechart.chart.ChartUtils;
import org.simantics.jfreechart.chart.properties.AllVariablesOfModel;
import org.simantics.jfreechart.chart.properties.ChartVariable;
import org.simantics.jfreechart.chart.properties.LabelPropertyTabContributor;
import org.simantics.jfreechart.chart.properties.xyline.AxisAndVariablesExplorerComposite;
import org.simantics.layer0.Layer0;
import org.simantics.sysdyn.JFreeChartResource;
import org.simantics.utils.datastructures.ArrayMap;
import org.simantics.utils.ui.AdaptionUtils;
import org.simantics.utils.ui.ExceptionUtils;

/**
 * Tab for modifying series in a pie chart configuration
 * @author Teemu Lempinen
 *
 */
public class PieSeriesTab2 extends LabelPropertyTabContributor implements Widget {

    private GraphExplorerComposite explorer;
    private ScrolledComposite propertyContainer;
    private WidgetSupportImpl additionalSupport;
    private Button add, remove;
    private Resource chartResource;
    private int options;

    public PieSeriesTab2(Object id, int options) {
        super(id);
        additionalSupport = new WidgetSupportImpl();
        this.options = options;
    }

    @Override
    public void createControls(Composite body, IWorkbenchSite site, final ISessionContext context, WidgetSupport support) {
        support.register(this);
        Composite composite = new Composite(body, SWT.NONE);
        GridDataFactory.fillDefaults().grab(true, true).applyTo(composite);
        GridLayoutFactory.fillDefaults().numColumns(2).margins(3, 3).applyTo(composite);

        // (Ontology-based) GraphExplorer displaying variables of a pie chart
        explorer = new AxisAndVariablesExplorerComposite(ArrayMap.keys(
                "displaySelectors", "displayFilter").values(false, false), site, composite, support, SWT.FULL_SELECTION | SWT.BORDER | SWT.SINGLE);
        explorer.setBrowseContexts(JFreeChartResource.URIs.PieSeriesBrowseContext);
        explorer.setInputSource(new SingleSelectionInputSource(
                Resource.class));
        explorer.getExplorer().setAutoExpandLevel(2); // Expand everything in the beginning
        explorer.finish();

        ((Tree)explorer.getExplorerControl()).addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                updateSelection(context);
            }
        });
        GridDataFactory.fillDefaults().hint(250, SWT.DEFAULT).grab(false, true).applyTo(explorer);

        // Scrolled composite for displaying properties of a selection in explorer
        propertyContainer = new ScrolledComposite(composite, SWT.H_SCROLL | SWT.V_SCROLL);
        GridDataFactory.fillDefaults().span(1, 2).grab(true, true).applyTo(propertyContainer);
        GridLayoutFactory.fillDefaults().applyTo(propertyContainer);
        propertyContainer.setExpandHorizontal(true);
        propertyContainer.setExpandVertical(true);


        // Buttons for adding and removing variables from a pie plot
        Composite buttonComposite = new Composite(composite, SWT.NONE);
        GridDataFactory.fillDefaults().applyTo(buttonComposite);
        GridLayoutFactory.fillDefaults().numColumns(3).applyTo(buttonComposite);

        add = new Button(buttonComposite, additionalSupport, SWT.NONE);
        add.setText("Add");
        add.addSelectionListener(new NewVariableListener(context));

        remove = new Button(buttonComposite, additionalSupport, SWT.NONE);
        remove.setText("Remove");
        remove.addSelectionListener(new RemoveListener(context));
    }

    /**
     * Updates the content of propertyContainer  
     * @param context
     */
    private void updateSelection(ISessionContext context) {
        ISelectionProvider selectionProvider = (ISelectionProvider)explorer.getAdapter(ISelectionProvider.class);
        IStructuredSelection selection = (IStructuredSelection)selectionProvider.getSelection();
        final Resource resource = AdaptionUtils.adaptToSingle(selection, Resource.class);
        if(resource == null)
            return;

        for(Control child : propertyContainer.getChildren()) {
            child.dispose();
        }

        try {
        	AllVariablesOfModel query = AllVariablesOfModel.withRandomResource(context, resource);
        	Collection<ChartVariable> variables = context.getSession().syncRequest(query);
	        PieSeriesPropertyComposite2 spc = new PieSeriesPropertyComposite2(propertyContainer, context, additionalSupport, variables, options, SWT.NONE);
	        propertyContainer.setContent(spc);
	        Point size = spc.computeSize(SWT.DEFAULT, SWT.DEFAULT);
	        propertyContainer.setMinSize(size);
	
	        additionalSupport.fireInput(context, selection);
        } catch (DatabaseException e) {
        	ExceptionUtils.logAndShowError(e);
        }
    }


    /**
     * SelectionListener for adding a new variable to a plot
     * @author Teemu Lempinen
     *
     */
    private class NewVariableListener extends SelectionListenerImpl<Resource> {

        public NewVariableListener(ISessionContext context) {
            super(context);
        }

        @Override
        public void apply(WriteGraph graph, Resource input) throws DatabaseException {
            JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
            Layer0 l0 = Layer0.getInstance(graph);
            Resource dataset = null;
            if(input == null) {
                if(chartResource != null) {
                    Resource plot = graph.syncRequest(new PossibleObjectWithType(chartResource, l0.ConsistsOf, jfree.Plot));
                    if(plot != null)
                        dataset = graph.syncRequest(new PossibleObjectWithType(plot, l0.ConsistsOf, jfree.Dataset));
                }
            } else {
                if(graph.isInstanceOf(input, jfree.Series)) {
                    dataset = graph.getPossibleObject(input, l0.PartOf);
                }
            }
            if(dataset != null) {
                // Create series with no rvi
                Resource series = ChartUtils.createSeries(graph, dataset, null);
                graph.claimLiteral(series, jfree.Series_exploded, false);
            }
        }
    }

    /**
     * SelectionListener for remove button
     * @author Teemu Lempinen
     *
     */
    private class RemoveListener extends SelectionListenerImpl<Resource> {

        public RemoveListener(ISessionContext context) {
            super(context);
        }

        /**
         * Removes selected resource from explorer
         */
        @Override
        public void apply(WriteGraph graph, Resource input) throws DatabaseException {
            if(input == null)
                return; 

            JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
            Layer0 l0 = Layer0.getInstance(graph);
            Resource list = null;
            if(graph.isInstanceOf(input, jfree.Series)) {
                // Remove series from dataset and seriesList
                Resource dataset = graph.getPossibleObject(input, l0.PartOf);
                if(dataset != null)
                    list = graph.getPossibleObject(dataset, jfree.Dataset_seriesList);

                if(list != null)
                    ListUtils.removeElement(graph, list, input);
                RemoverUtil.remove(graph, input);
            }
        }
    }
    
    @Override
    public void setInput(ISessionContext context, Object input) {
        chartResource = AdaptionUtils.adaptToSingle(input, Resource.class);
    }

}
