package org.simantics.jfreechart.chart.properties;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.ListenerList;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.resource.LocalResourceManager;
import org.eclipse.jface.resource.ResourceManager;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.MouseTrackListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Text;
import org.simantics.browsing.ui.swt.widgets.DefaultColorProvider;
import org.simantics.browsing.ui.swt.widgets.impl.ITrackedColorProvider;
import org.simantics.browsing.ui.swt.widgets.impl.ReadFactory;
import org.simantics.browsing.ui.swt.widgets.impl.Widget;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupport;
import org.simantics.db.management.ISessionContext;
import org.simantics.db.procedure.Listener;

/**
 * Widget for choosing and labeled object from a set of objects.
 * 
 * Supports cases when multiple objects have the same label (as much as possible)
 * 
 * 
 * Based on org.simantics.browsing.ui.swt.widgets.Trackedtext
 * 
 * 
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 *
 */
public class StringChooser<T> implements Widget{
    private static final int      EDITING                 = 1 << 0;
    private static final int      MODIFIED_DURING_EDITING = 1 << 1;

    /**
     * Used to tell whether or not a mouseDown has occurred after a focusGained
     * event to be able to select the whole text field when it is pressed for
     * the first time while the widget holds focus.
     */
    private static final int      MOUSE_DOWN_FIRST_TIME   = 1 << 2;
    private static final int      MOUSE_INSIDE_CONTROL    = 1 << 3;
    
    private int                   caretPositionBeforeEdit;
    
    private String                textBeforeEdit;

    private int                   state;

	private Map<T,String> objectToLabel;
	private Set<String> allowedLabels;
	
	private T selected;
	
    private final Display         display;

    private final Text            text;
    
    private CompositeListener     listener;

    private ListenerList          modifyListeners;

    private ReadFactory<?, T>     objectFactory;
    
    private ITrackedColorProvider colorProvider;

    private final ResourceManager resourceManager;
    
	private boolean moveCaretAfterEdit = true;
	
	private boolean selectAllOnStartEdit = true;
	
	public StringChooser(Composite parent, WidgetSupport support, int style) {
		this.state = 0;
        this.text = new Text(parent, style);
        this.display = text.getDisplay();
        this.resourceManager = new LocalResourceManager(JFaceResources.getResources(), text);
        this.colorProvider = new DefaultColorProvider(resourceManager);
        if (support!=null) support.register(this);
        initialize();
	}
	

	
	public ResourceManager getResourceManager() {
    	return resourceManager;
    }
	
	public void setFont(Font font) {
		text.setFont(font);
	}
	
	public void setObjectFactory(ReadFactory<?, T> objectFactory) {
		this.objectFactory = objectFactory;
	}
	
	public void setMoveCaretAfterEdit(boolean value) {
		this.moveCaretAfterEdit = value;
	}
	
	public void setData(Map<T,String> data) {
		this.objectToLabel = data;
		this.allowedLabels = new HashSet<String>();
		this.allowedLabels.addAll(objectToLabel.values());
	}
	
	public void setData(Collection<T> data) {
		this.objectToLabel = new HashMap<T, String>();
		this.allowedLabels = new HashSet<String>();
		for (T t : data) {
			String label = t.toString();
			objectToLabel.put(t, label);
			allowedLabels.add(label);
		}
	}
	
	public void setSelected(T selected) {
		if (selected != null) {
			String label = objectToLabel.get(selected);
			if (label == null)
				return;
			this.selected = selected;
			this.text.setText(label);
		} else {
			this.selected = null;
			this.text.setText("");
		}
	}
	
	public void setSelected(String label) {
		// TODO : we could create a label to object map.
		for (T t : objectToLabel.keySet()) {
			if (label.equals(objectToLabel.get(t))) {
				setSelected(t);
				return;
			}
		}
	}
	
	
	@Override
	public void setInput(ISessionContext context, Object input) {
        if (modifyListeners != null) {
            for (Object o : modifyListeners.getListeners()) {
            	if(o instanceof Widget) {
                    ((Widget) o).setInput(context, input);
            	}
            }
        }
		
		if(objectFactory != null) {
			objectFactory.listen(context, input, new Listener<T>() {

				@Override
                public void exception(final Throwable t) {
					display.asyncExec(new Runnable() {

						@Override
						public void run() {
							if(isDisposed()) return;
//							System.out.println("Button received new text: " + text);
							text.setText(t.toString());
						}

					});
				}

				@Override
				public void execute(final T object) {
					
					if(text.isDisposed()) return;
					
					display.asyncExec(new Runnable() {

						@Override
						public void run() {
							if(isDisposed()) return;
							setSelected(object);
//							text.getParent().layout();
//							text.getParent().getParent().layout();
						}

					});
				}

				@Override
				public boolean isDisposed() {
					return text.isDisposed();
				}

			});
		}
		
	}
	
	/**
     * Common initialization. Assumes that text is already created.
     */
    private void initialize() {
        Assert.isNotNull(text);

        text.setBackground(colorProvider.getInactiveBackground());
        text.setDoubleClickEnabled(false);

        listener = new CompositeListener();

        text.addModifyListener(listener);
        text.addDisposeListener(listener);
        text.addKeyListener(listener);
        text.addMouseTrackListener(listener);
        text.addMouseListener(listener);
        text.addFocusListener(listener);
    }
    
    public void startEdit(boolean selectAll) {
        if (isEditing()) {
            // Print some debug incase we end are in an invalid state
            System.out.println("TrackedText: BUG: startEdit called when in editing state");
        }
//        System.out.println("start edit: selectall=" + selectAll + ", text=" + text.getText() + ", caretpos=" + caretPositionBeforeEdit);

        // Backup text-field data for reverting purposes
        caretPositionBeforeEdit = text.getCaretPosition();
        textBeforeEdit = text.getText();

        // Signal editing state
        setBackground(colorProvider.getEditingBackground());

        if (selectAll) {
            text.selectAll();
        }
        state |= EDITING | MOUSE_DOWN_FIRST_TIME;
    }

    @SuppressWarnings("unchecked")
	private void applyEdit() {
        try {
            if (isTextValid() != null) {
                text.setText(textBeforeEdit);
            } else if (isModified() && !text.getText().equals(textBeforeEdit)) {
            	setSelected(text.getText());
                //System.out.println("apply");
                if (modifyListeners != null) {
                    StringChooserModifyEvent<T> event = new StringChooserModifyEvent<T>(text, selected, text.getText());
                    for (Object o : modifyListeners.getListeners()) {
                        ((StringChooserModifyListener<T>) o).modifySelection(event);
                    }
                }
            }
        } catch (Throwable t) {
            t.printStackTrace();
        } finally {
            endEdit();
        }
    }

    private void endEdit() {
        if (text.isDisposed())
            return;

        if (!isEditing()) {
            // Print some debug incase we end are in an invalid state
            //ExceptionUtils.logError(new Exception("BUG: endEdit called when not in editing state"));
            //System.out.println();
        }
        setBackground(isMouseInsideControl() ? colorProvider.getHoverBackground() : colorProvider.getInactiveBackground());
//        System.out.println("endEdit: " + text.getText() + ", caret: " + text.getCaretLocation() + ", selection: " + text.getSelection());
        // Always move the caret to the end of the string
        if(moveCaretAfterEdit)
            text.setSelection(text.getCharCount());
        state &= ~(EDITING | MOUSE_DOWN_FIRST_TIME);
        setModified(false);
    }

    private void revertEdit() {
        if (!isEditing()) {
            // Print some debug incase we end are in an invalid state
            //ExceptionUtils.logError(new Exception("BUG: revertEdit called when not in editing state"));
            System.out.println("BUG: revertEdit called when not in editing state");
        }
        text.setText(textBeforeEdit);
        text.setSelection(caretPositionBeforeEdit);
        setBackground(isMouseInsideControl() ? colorProvider.getHoverBackground() : colorProvider.getInactiveBackground());
        state &= ~(EDITING | MOUSE_DOWN_FIRST_TIME);
        setModified(false);
    }

    private boolean isEditing() {
        return (state & EDITING) != 0;
    }

    private void setModified(boolean modified) {
        if (modified) {
            state |= MODIFIED_DURING_EDITING;
        } else {
            state &= ~MODIFIED_DURING_EDITING;
        }
    }

    private boolean isMouseInsideControl() {
        return (state & MOUSE_INSIDE_CONTROL) != 0;
    }

    private void setMouseInsideControl(boolean inside) {
        if (inside)
            state |= MOUSE_INSIDE_CONTROL;
        else
            state &= ~MOUSE_INSIDE_CONTROL;
    }

    private boolean isModified() {
        return (state & MODIFIED_DURING_EDITING) != 0;
    }

    public void setSelectAllOnStartEdit(boolean selectAll) {
        this.selectAllOnStartEdit = selectAll;
    }
    
    public void setEditable(boolean editable) {
        if (editable) {
            text.setEditable(true);
            setBackground(isMouseInsideControl() ? colorProvider.getHoverBackground() : colorProvider.getInactiveBackground());
        } else {
            text.setEditable(false);
            text.setBackground(null);
        }
    }

    public void setText(String text) {
        this.text.setText(text);
    }

    public void setTextWithoutNotify(String text) {
        this.text.removeModifyListener(listener);
        setText(text);
        this.text.addModifyListener(listener);
    }

    public Text getWidget() {
        return text;
    }

	public synchronized void addModifyListener(StringChooserModifyListener<T> listener) {
        if (modifyListeners == null) {
            modifyListeners = new ListenerList(ListenerList.IDENTITY);
        }
        modifyListeners.add(listener);
    }

	public synchronized void removeModifyListener(StringChooserModifyListener<T> listener) {
        if (modifyListeners == null)
            return;
        modifyListeners.remove(listener);
    }

  

    private String isTextValid() {
        if (allowedLabels.contains(getWidget().getText()))
        	return null;
        return "There is no such object.";
    }

    public void setColorProvider(ITrackedColorProvider provider) {
        Assert.isNotNull(provider);
        this.colorProvider = provider;
    }

    private void setBackground(Color background) {
    	if(text.isDisposed()) return;
        if (!text.getEditable()) {
            // Do not alter background when the widget is not editable.
            return;
        }
        text.setBackground(background);
    }
    
    public boolean isDisposed() {
    	return text.isDisposed();
    }
    
    public Display getDisplay() {
    	return display;
    }
    
    public String getText() {
		return text.getText();
	}
    
    public int getCaretPosition() {
    	return text.getCaretPosition();
    }
	
	
    /**
     * A composite of many UI listeners for creating the functionality of this
     * class.
     */
    private class CompositeListener
    implements ModifyListener, DisposeListener, KeyListener, MouseTrackListener,
    MouseListener, FocusListener
    {
        // Keyboard/editing events come in the following order:
        //   1. keyPressed
        //   2. verifyText
        //   3. modifyText
        //   4. keyReleased

        @Override
        public void modifyText(ModifyEvent e) {
            //System.out.println("modifyText: " + e);
            setModified(true);

            String valid = isTextValid();
            if (valid != null) {
                setBackground(colorProvider.getInvalidBackground());
            } else {
                if (isEditing())
                    setBackground(colorProvider.getEditingBackground());
                else
                    setBackground(colorProvider.getInactiveBackground());
            }
        }

        @Override
        public void widgetDisposed(DisposeEvent e) {
            getWidget().removeModifyListener(this);
        }

        private boolean isMultiLine() {
            return (text.getStyle() & SWT.MULTI) != 0;
        }

        private boolean hasMultiLineCommitModifier(KeyEvent e) {
            return (e.stateMask & SWT.CTRL) != 0;
        }

        @Override
        public void keyPressed(KeyEvent e) {
            //System.out.println("keyPressed: " + e);
            if (!isEditing()) {
                // ESC, ENTER & keypad ENTER must not start editing
                if (e.keyCode == SWT.ESC)
                    return;

                if (!isMultiLine()) {
                    if (e.keyCode == SWT.F2 || e.keyCode == SWT.CR || e.keyCode == SWT.KEYPAD_CR) {
                        startEdit(selectAllOnStartEdit);
                    } else if (e.character != '\0') {
                        startEdit(false);
                    }
                } else {
                    // In multi-line mode, TAB must not start editing!
                    if (e.keyCode == SWT.F2) {
                        startEdit(selectAllOnStartEdit);
                    } else if (e.keyCode == SWT.CR || e.keyCode == SWT.KEYPAD_CR) {
                        if (hasMultiLineCommitModifier(e)) {
                            e.doit = false;
                        } else {
                            startEdit(false);
                        }
                    } else if (e.keyCode == SWT.TAB) {
                        text.traverse(((e.stateMask & SWT.SHIFT) != 0) ? SWT.TRAVERSE_TAB_PREVIOUS : SWT.TRAVERSE_TAB_NEXT);
                        e.doit = false;
                    } else if (e.character != '\0') {
                        startEdit(false);
                    }
                }
            } else {
                // ESC reverts any changes made during this edit
                if (e.keyCode == SWT.ESC) {
                    revertEdit();
                }
                if (!isMultiLine()) {
                    if (e.keyCode == SWT.CR || e.keyCode == SWT.KEYPAD_CR) {
                        applyEdit();
                    }
                } else {
                    if (e.keyCode == SWT.CR || e.keyCode == SWT.KEYPAD_CR) {
                        if (hasMultiLineCommitModifier(e)) {
                            applyEdit();
                            e.doit = false;
                        }
                    }
                }
            }
        }

        @Override
        public void keyReleased(KeyEvent e) {
            //System.out.println("keyReleased: " + e);
        }

        @Override
        public void mouseEnter(MouseEvent e) {
            //System.out.println("mouseEnter");
            if (!isEditing()) {
                setBackground(colorProvider.getHoverBackground());
            }
            setMouseInsideControl(true);
        }

        @Override
        public void mouseExit(MouseEvent e) {
            //System.out.println("mouseExit");
            if (!isEditing()) {
                setBackground(colorProvider.getInactiveBackground());
            }
            setMouseInsideControl(false);
        }

        @Override
        public void mouseHover(MouseEvent e) {
            //System.out.println("mouseHover");
            setMouseInsideControl(true);
        }

        @Override
        public void mouseDoubleClick(MouseEvent e) {
            //System.out.println("mouseDoubleClick: " + e);
            if (e.button == 1) {
                getWidget().selectAll();
            }
        }

        @Override
        public void mouseDown(MouseEvent e) {
            //System.out.println("mouseDown: " + e);
            if (!isEditing()) {
                // In reality we should never get here, since focusGained
                // always comes before mouseDown, but let's keep this
                // fallback just to be safe.
                if (e.button == 1) {
                    startEdit(selectAllOnStartEdit);
                }
            } else {
                if (e.button == 1 && (state & MOUSE_DOWN_FIRST_TIME) != 0) {
                    if (!isMultiLine()) {
                        // This is useless for multi-line texts
                        getWidget().selectAll();
                    }
                    state &= ~MOUSE_DOWN_FIRST_TIME;
                }
            }
        }

        @Override
        public void mouseUp(MouseEvent e) {
        }

        @Override
        public void focusGained(FocusEvent e) {
            //System.out.println("focusGained");
            if (!isEditing()) {
                if (!isMultiLine()) {
                    // Always start edit on single line texts when focus is gained
                    startEdit(selectAllOnStartEdit);
                }
            }
        }

        @Override
        public void focusLost(FocusEvent e) {
            //System.out.println("focusLost");
            if (isEditing()) {
                applyEdit();
            }
        }
    }
    
    public void dispose() {
    	allowedLabels.clear();
    	allowedLabels = null;
    	objectFactory = null;
    	objectToLabel.clear();
    	objectToLabel = null;
    	
    }
    
    
}
