/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart.properties;

import org.simantics.browsing.ui.swt.widgets.impl.ReadFactoryImpl;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.exception.DatabaseException;
import org.simantics.layer0.Layer0;
import org.simantics.utils.datastructures.Quad;

/**
 * PropertyFactory for finding a double property. Supports also finding the 
 * property from a first occurrence of resource ConsistsOf type HasProperty   
 * 
 * @author Teemu Lempinen
 * @author Marko Luukkainen
 *
 */
public class DoublePropertyFactory2 extends ReadFactoryImpl<Resource, String> {

    final private String propertyURI;
    final private String typeURI;
    final private Double defaultValue;
    
    /**
     * PropertyFactory for finding a boolean property with propertyURI
     * 
     * @param propertyURI URI for the boolean property
     */
    public DoublePropertyFactory2(String propertyURI) {
        this(null, propertyURI);
    }
    

    /**
     * PropertyFactory for finding a boolean property with propertyURI.
     *  
     * Finds the property for first ObjectWithType(resource, L0.ConsistsOf, type)
     *  
     * Supports inverting the result (e.g. if required information is IsHidden, but database contains IsVisible)
     *  
     * @param typeURI URI for a resource (resource ConsistsOf type) (null allowed)
     * @param propertyURI URI for the boolean property
     * @param inverse Invert the result?
     */
    public DoublePropertyFactory2(String typeURI, String propertyURI) {
        this(typeURI, propertyURI, 0.0);
    }
    
    /**
     * PropertyFactory for finding a boolean property with propertyURI.
     *  
     * Finds the property for first ObjectWithType(resource, L0.ConsistsOf, type)
     *  
     * Supports inverting the result (e.g. if required information is IsHidden, but database contains IsVisible)
     * 
     * @param typeURI URI for a resource (resource ConsistsOf type) (null allowed -> not used)
     * @param propertyURI URI for the boolean property
     * @param inverse Invert the result?
     * @param defaultValue default value
     */
    public DoublePropertyFactory2(String typeURI, String propertyURI, double defaultValue) {
        this.propertyURI = propertyURI;
        this.typeURI = typeURI;
        this.defaultValue = defaultValue;
    }

    @Override
    public Object getIdentity(Object inputContents) {
        return new Quad<Resource, String, Object, Double>((Resource) inputContents, propertyURI, getClass(), defaultValue);
    }

    @Override
    public String perform(ReadGraph graph, Resource r) throws DatabaseException {
        if(typeURI == null) {
            // if no typeUri, use the default resource r
            return getValue(graph, r);
        } else {
            // typeURI was defined, find the property for the first occurence of ConsistsOf type
            Resource type =  graph.getResource(typeURI);
            for(Resource o : graph.syncRequest(new ObjectsWithType(r, Layer0.getInstance(graph).ConsistsOf, type))) {
                // Returns the value for the first occurrence
                return getValue(graph, o);
            }
        }
        // if nothing was found with typeURI
        return "";
    }
    
    /**
     * Return the value for a Boolean literal possibly inverted (or default if resource != Boolean literal) 
     * 
     * @param graph ReadGraph
     * @param resource Literal Boolean resource 
     * @return value of the parameter (or default or inverted)
     * @throws DatabaseException
     */
    private String getValue(ReadGraph graph, Resource resource) throws DatabaseException {
        Double value = graph.getPossibleRelatedValue(resource, graph.getResource(propertyURI), Bindings.DOUBLE);
        if(value != null) {
            return value.toString();
        } else if (defaultValue != null){
            return defaultValue.toString();
        }
        return "";
    }
}