/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart.properties;

import java.util.Arrays;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.resource.LocalResourceManager;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.widgets.ColorDialog;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.simantics.Simantics;
import org.simantics.browsing.ui.swt.widgets.Button;
import org.simantics.browsing.ui.swt.widgets.impl.ReadFactoryImpl;
import org.simantics.browsing.ui.swt.widgets.impl.SelectionListenerImpl;
import org.simantics.browsing.ui.swt.widgets.impl.Widget;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupport;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.db.management.ISessionContext;
import org.simantics.db.procedure.Listener;
import org.simantics.db.request.Read;
import org.simantics.diagram.stubs.G2DResource;
import org.simantics.layer0.Layer0;
import org.simantics.sysdyn.JFreeChartResource;
import org.simantics.utils.RunnableWithObject;
import org.simantics.utils.datastructures.Triple;
import org.simantics.utils.ui.AdaptionUtils;
import org.simantics.utils.ui.gfx.ColorImageDescriptor;

/**
 * Composite for selecting a color for a chart component
 * 
 * @author Teemu Lempinen
 *
 */
public class ColorPicker extends Composite implements Widget {

    Button defaultColor, customColor, color;
    
    /**
     * Create a composite containing radio buttons for default or custom color. Color chooser button is active
     * when the custom radio button is selected. Color chooser uses {@link ColorDialog} to select a color 
     * 
     * @param parent Composite
     * @param context ISessionContext
     * @param support WidgetSupport
     * @param style SWT style
     */
    public ColorPicker(Composite parent, ISessionContext context, WidgetSupport support, int style) {
        this(parent, context, support, style, true);
    }

    /**
     * Create a composite containing radio buttons for default or custom color. Color chooser button is active
     * when the custom radio button is selected. Color chooser uses {@link ColorDialog} to select a color 
     * 
     * @param parent Composite
     * @param context ISessionContext
     * @param support WidgetSupport
     * @param style SWT style
     * @param defaultColor provide default color widget?
     */
    public ColorPicker(Composite parent, ISessionContext context, WidgetSupport support, int style, boolean defaultColor) {
        super(parent, style);
        support.register(this);
        
        if(support.getParameter(WidgetSupport.RESOURCE_MANAGER) == null) {
            LocalResourceManager resourceManager = new LocalResourceManager(JFaceResources.getResources(), this);
            support.setParameter(WidgetSupport.RESOURCE_MANAGER, resourceManager);
        }
        
        if(defaultColor) {
            GridLayoutFactory.fillDefaults().numColumns(4).applyTo(this);

            this.defaultColor = new Button(this, support, SWT.RADIO);
            this.defaultColor.setText("default");
            this.defaultColor.setSelectionFactory(new DefaultColorSelectionFactory(false));
            this.defaultColor.addSelectionListener(new DefaultColorSelectionListener(context));
            GridDataFactory.fillDefaults().applyTo(this.defaultColor.getWidget());

            customColor = new Button(this, support, SWT.RADIO);
            customColor.setText("custom");
            customColor.setSelectionFactory(new DefaultColorSelectionFactory(true));
            customColor.addSelectionListener(new DefaultColorSelectionListener(context));

            GridDataFactory.fillDefaults().applyTo(customColor.getWidget());
        } else {
            GridLayoutFactory.fillDefaults().applyTo(this);
        }
        
        
        color = new Button(this, support, SWT.PUSH);
        color.setImageFactory(new ColorImageFactoryFactory(this));
        color.addSelectionListener(new ColorSelectionListener(context));
        color.getWidget().setEnabled(false);
        GridDataFactory.fillDefaults().applyTo(color.getWidget());
    }

    /**
     * Method for finding the resource for which the color is selected. 
     * 
     * @param graph ReadGraph
     * @param input input from WidgetSupport
     * @return
     * @throws DatabaseException
     */
    protected Resource getResource(ReadGraph graph, Resource input) throws DatabaseException {
        return input;
    }
    
    /**
     * Method for getting the relation with which the g2d.Color is related to a resource
     * 
     * @param graph ReadGraph
     * @return Color relation
     * @throws DatabaseException
     */
    protected Resource getColorRelation(ReadGraph graph) throws DatabaseException {
        JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
        return jfree.color;
    }
    
    
    @Override
    public void setInput(ISessionContext context, Object input) {
        final Resource resource = AdaptionUtils.adaptToSingle(input, Resource.class);
        
        // Create a listener to define the enabled state of the color chooser button
        context.getSession().asyncRequest(new Read<Float[]>() {

            @Override
            public Float[] perform(ReadGraph graph) throws DatabaseException {
                if(graph.hasStatement(getResource(graph, resource), getColorRelation(graph))) {
                    float[] components = graph.getPossibleRelatedValue(getResource(graph, resource), getColorRelation(graph));
                    Float[] result = new Float[components.length];
                    for(int i = 0; i < components.length; i++) result[i] = components[i];
                    return result;
                } else {
                    return null;
                }
            }
            
        }, new Listener<Float[]>() {

            @Override
            public void execute(Float[] result) {
                if(!color.getWidget().isDisposed()) {
                    color.getWidget().getDisplay().asyncExec(new RunnableWithObject(result != null ? true : false) {
                        @Override
                        public void run() {
                            if(!color.getWidget().isDisposed()) {
                                if(!Boolean.TRUE.equals(getObject()))
                                    color.getWidget().setEnabled(false);
                                else
                                    color.getWidget().setEnabled(true);
                            }
                        }
                    });
                }
            }

            @Override
            public void exception(Throwable t) {
                t.printStackTrace();
            }

            @Override
            public boolean isDisposed() {
                return color.getWidget().isDisposed();
            }
        });
    }
    

    /**
     * ImageFactory returning an image for color button
     * @author Teemu Lempinen
     *
     */
    private class ColorImageFactoryFactory extends ReadFactoryImpl<Resource, ImageDescriptor> {
        
        private ColorPicker picker;
        
        public ColorImageFactoryFactory(ColorPicker colorPicker) {
            this.picker = colorPicker;
        }

        @Override
        public ImageDescriptor perform(ReadGraph graph, Resource input) throws DatabaseException {
            RGB rgb = getColor(graph, getResource(graph, input));
            return new ColorImageDescriptor(rgb.red, rgb.green, rgb.blue, 20, 20, false);
 
        }
        
        @Override
        public Object getIdentity(Object inputContents) {
            return new Triple<Object, ColorPicker, Class<?>>(inputContents, picker, getClass());
        }

    }

    /**
     * Get RGB from a color literal resource. If resource is not a color resource, return blue (RGB 0, 0, 255)
     * @param graph ReadGraph
     * @param input Color literal resource (float[4])
     * @return RGB color
     * @throws DatabaseException
     */
    private RGB getColor(ReadGraph graph, Resource input) throws DatabaseException{
        float[] colorComponents = graph.getPossibleRelatedValue(input, getColorRelation(graph));
        RGB rgb;
        if(colorComponents == null)
            rgb = new RGB(0, 0, 255);
        else
            rgb = new RGB((int)(colorComponents[0] * 255.0f), 
                    (int)(colorComponents[1] * 255.0f), 
                    (int)(colorComponents[2] * 255.0f));
        return rgb;
    }


    /**
     * SelectionListener for color button. 
     * 
     * @author Teemu Lempinen
     *
     */
    private class ColorSelectionListener extends SelectionListenerImpl<Resource> {

        private SelectionEvent e;
        private RGB rgb;
        
        /**
         * 
         * @param context ISessionContext
         */
        public ColorSelectionListener(ISessionContext context) {
            super(context);
        }

        @Override
        public void widgetSelected(SelectionEvent e) {
            if(color.getWidget().isDisposed())
                return;
            // Save the event for coordinates
            this.e = e;
            super.widgetSelected(e);
        }
        
        @Override
        public void beforeApply(final Resource input) {
            try {
                final RGB oldRGB = Simantics.getSession().syncRequest(new UniqueRead<RGB>() {

                    @Override
                    public RGB perform(ReadGraph graph) throws DatabaseException {
                        Resource resource = getResource(graph, input);
                        return getColor(graph, resource);
                    }
                });
                
                final Display display = color.getWidget().getDisplay();
                
                // Use color dialog to select a color
                Shell shell = new Shell(display);
                ColorDialog cd = new ColorDialog(shell);
                Point point = color.getWidget().toDisplay(e.x - 150, e.y - 150);
                cd.getParent().setLocation(point.x, point.y);
                cd.setText("Select color");
                cd.setRGB(oldRGB);
                rgb = cd.open();
                if(rgb == null)
                    return;
                
                
            } catch (DatabaseException e) {
                e.printStackTrace();
            }
        }

        @Override
        public void apply(WriteGraph graph, Resource input) throws DatabaseException {
            
            G2DResource g2d = G2DResource.getInstance(graph);
            float[] components = new float[] {rgb.red / 255.0f,  rgb.green / 255.0f, rgb.blue / 255.0f, 1.0f};
            Resource resource = getResource(graph, input);
            float[] currentComponents = graph.getPossibleRelatedValue2(resource, getColorRelation(graph));
            if (currentComponents == null || !Arrays.equals(components, currentComponents)) {
                graph.claimLiteral(resource, getColorRelation(graph), g2d.Color, components);
                Layer0Utils.addCommentMetadata(graph, "Modified color of " + NameUtils.getSafeName(graph, resource) + " to " + Arrays.toString(components));
            }
        }
    }

    /**
     * SelectionFactory for default and custom color radio buttons
     * @author Teemu Lempinen
     *
     */
    private class DefaultColorSelectionFactory extends ReadFactoryImpl<Resource, Boolean> {

        private final Boolean isCustom;

        /**
         * 
         * @param isCustom Is this custom button?
         */
        public DefaultColorSelectionFactory(Boolean isCustom) {
            super();
            this.isCustom = isCustom;
        }

        @Override
        public Object getIdentity(Object inputContents) {
            return new Triple<Resource, Object, Boolean>((Resource) inputContents, getClass(), isCustom);
        }

        @Override
        public Boolean perform(ReadGraph graph, Resource input) throws DatabaseException {
            Resource r = graph.getPossibleObject(getResource(graph, input), getColorRelation(graph));
            boolean result = false; // Default == not selected
            if(r == null && !isCustom) {
                // No color definition and default-button -> selected
                result =  true;
            } else if(r != null && isCustom) {
                // color definition and custom button -> selected
                result =  true;
            }
            return result;
        }

    }

    /**
     * SelectionListener for default and custom radio buttons
     * 
     * @author Teemu Lempinen
     *
     */
    private class DefaultColorSelectionListener extends SelectionListenerImpl<Resource> {

        private SelectionEvent e;

        public DefaultColorSelectionListener(ISessionContext context) {
            super(context);
        }

        @Override
        public void widgetSelected(SelectionEvent e) {
            this.e = e; 
            super.widgetSelected(e);
        }

        @Override
        public void apply(WriteGraph graph, Resource input) throws DatabaseException {
            Resource resource = getResource(graph, input);
            if(customColor.getWidget().equals(e.widget)) {
                // Custom selected. If there is no color already, create a default Blue color
                G2DResource g2d = G2DResource.getInstance(graph);
                if(graph.getPossibleObject(resource, getColorRelation(graph)) == null) {
                    float[] components = java.awt.Color.BLUE.getColorComponents(new float[4]);
                    components[3] = 1.0f;
                    graph.claimLiteral(resource, getColorRelation(graph), g2d.Color, components);
                    Layer0Utils.addCommentMetadata(graph, "Color for " + graph.getPossibleRelatedValue2(resource, Layer0.getInstance(graph).HasName, Bindings.STRING) + " set to custom ");
                }
            } else {
                // Default selected, remove color definition
                Object o = graph.getPossibleObject(resource, getColorRelation(graph));
                if (o != null) {
                    graph.deny(resource, getColorRelation(graph));
                    Layer0Utils.addCommentMetadata(graph, "Color for " + graph.getPossibleRelatedValue2(resource, Layer0.getInstance(graph).HasName, Bindings.STRING) + " set to Default");
                }
            }
        }
    }


}
