/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart.properties;

import org.simantics.browsing.ui.swt.widgets.impl.SelectionListenerImpl;
import org.simantics.databoard.Bindings;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.db.management.ISessionContext;
import org.simantics.layer0.Layer0;

/**
 * Class for setting a boolean value when a selection occurs. (check box buttons)
 * 
 * @author Teemu Lempinen
 *
 */
public class BooleanSelectionListener extends SelectionListenerImpl<Resource> {

    final private String propertyURI;
    final private String typeUri;
    final private boolean defaultValue;
    
    /**
     * Boolean selection listener for property with propertyURI
     * 
     * @param context ISessionContext
     * @param propertyURI uri of the boolean property
     */
    public BooleanSelectionListener(ISessionContext context, String propertyURI) {
        this(context, null, propertyURI, false);
    }

    /**
     * Boolean selection listener for property with propertyURI
     * Sets the property for all ObjectWithType(resource, L0.ConsistsOf, type)
     * 
     * @param context ISessionContext
     * @param typeUri URI for a resource (resource ConsistsOf type) (null allowed -> not used)
     * @param propertyURI uri of the boolean property
     * @param defaultValue expected value if the property does not exist
     */
    public BooleanSelectionListener(ISessionContext context, String typeUri, String propertyURI, boolean defaultValue) {
        super(context);
        this.propertyURI = propertyURI;
        this.typeUri = typeUri;
        this.defaultValue = defaultValue;
    }
    
    public BooleanSelectionListener(ISessionContext context, String typeUri, String propertyURI) {
       this(context, typeUri, propertyURI, false);
    }


    @Override
    public void apply(WriteGraph graph, Resource chart) throws DatabaseException {
        if(typeUri == null) {
            setValue(graph, chart);
        } else {
            Resource type =  graph.getResource(typeUri);
            for(Resource object : graph.syncRequest(new ObjectsWithType(chart, Layer0.getInstance(graph).ConsistsOf, type))) {
                setValue(graph, object);
            }
        }
            
    }
    
    /**
     * Set boolean value for Boolean literal resource (inverts the current value).
     * @param graph ReadGraph
     * @param resource Boolean literal resource
     * @throws DatabaseException
     */
    private void setValue(WriteGraph graph, Resource resource) throws DatabaseException {
        Resource property =  graph.getResource(propertyURI);
        Boolean value = graph.getPossibleRelatedValue(resource, property, Bindings.BOOLEAN);
        if (value == null) {
        	graph.claimLiteral(resource, property, !defaultValue);
        	Layer0Utils.addCommentMetadata(graph, "Modified " + graph.getPossibleRelatedValue2(property, Layer0.getInstance(graph).HasName, Bindings.STRING) + " for " + graph.getPossibleRelatedValue2(resource, Layer0.getInstance(graph).HasLabel, Bindings.STRING) + " to " + !defaultValue);
        } else { 
            boolean newValue = Boolean.FALSE.equals(value);
        	graph.claimLiteral(resource, property, newValue);
            Layer0Utils.addCommentMetadata(graph, "Modified " + graph.getPossibleRelatedValue2(property, Layer0.getInstance(graph).HasName, Bindings.STRING) + " for " + graph.getPossibleRelatedValue2(resource, Layer0.getInstance(graph).HasLabel, Bindings.STRING) + " to " + newValue);
        }
    }
}