/*******************************************************************************
 * Copyright (c) 2007, 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart.element;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Stroke;
import java.awt.geom.AffineTransform;
import java.awt.geom.Line2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;

import javax.swing.JPanel;

import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.simantics.scenegraph.ISelectionPainterNode;
import org.simantics.scenegraph.g2d.events.EventTypes;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseButtonPressedEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseButtonReleasedEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseDragBegin;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseMovedEvent;
import org.simantics.scenegraph.swing.ComponentNode;
import org.simantics.scenegraph.utils.GeometryUtils;
import org.simantics.scenegraph.utils.NodeUtil;

/**
 * Chart node for displaying jfreechart charts in diagrams
 * @author Teemu Lempinen
 *
 */
public class ChartNode extends ComponentNode<JPanel> implements ISelectionPainterNode {

    private static final long serialVersionUID = 5013911689968911010L;
    private boolean hover = false;
    private boolean dragging = false;

    private ResizeListener resizeListener; 

    protected transient JFreeChart chart = null;

    public boolean dragging() {
        return dragging;
    }

    public void setChart(JFreeChart chart) {
        this.chart = chart;
    }

    public void setResizeListener(ResizeListener listener) {
        this.resizeListener = listener;
    }

    @SyncField({"hover"})
    public void setHover(boolean hover) {
        this.hover = hover;
        repaint();
    }

    @Override
    public void cleanup() {
        removeEventHandler(this);
        super.cleanup();
    }

    @Override
    public void init() {
        super.init();
    }

    public static void expand(Rectangle2D rectangle, double scaleX, double scaleY) {
        GeometryUtils.expandRectangle(rectangle,
                5 * scaleY > 5 ? 5 * scaleY : 5, 
                        3 * scaleY > 3 ? 3 * scaleY : 3, 
                                3 * scaleX > 3 ? 3 * scaleX : 3, 
                                        3 * scaleX > 3 ? 3 * scaleX : 3);
    }

    @Override
    public void render(Graphics2D g2d) {
        if (component == null && chart != null) {
            // Need the chart, so this cannot be done during initialization
            component = new ChartPanel(chart, false);
            ((ChartPanel)component).setRefreshBuffer(false);
            component.setIgnoreRepaint(true); 
            component.setDoubleBuffered(false);
            if(bounds != null) {
                component.setBounds(0, 0, 0, 0);
            }
            super.init();
            addEventHandler(this);
        }

        if (component != null) {
            AffineTransform ot = g2d.getTransform();
            g2d.transform(transform);
            double scaleX = g2d.getTransform().getScaleX();
            double scaleY = g2d.getTransform().getScaleY();

            AffineTransform at = new AffineTransform(transform);
            synchronized(transform) {
                at.setToTranslation(bounds.getMinX(), bounds.getMinY());
                at.scale(1/scaleX, 1/scaleY);
            }
            g2d.transform(at);
            int width = (int)(bounds.getWidth() * scaleX);
            int height = (int)(bounds.getHeight() * scaleY);

            if(hover || dragging) {
                // Borders
                Color orig = g2d.getColor();
                BasicStroke oldStroke = (BasicStroke) g2d.getStroke();
                g2d.setColor(Color.LIGHT_GRAY);
                Rectangle2D b = new Rectangle2D.Double(0, 0, width, height);
                Rectangle2D r = b.getBounds2D();
                expand(r, 1 * scaleX, 1 * scaleY);
                g2d.fill(r);

                // Lower right corner for dragging
                double x = r.getMaxX() - (0.33 * scaleX);
                double y = r.getMaxY() - (0.33 * scaleY);

                if(r.getMaxX() - x > 0.4) {                
                    // if there is enough space, use decorated corner
                    BasicStroke hilightStroke = new BasicStroke(oldStroke.getLineWidth() * 3);
                    for(int i = 1; i < 4; i++) {
                        Line2D line = new Line2D.Double(
                                x - (i * scaleX), 
                                y, 
                                x, 
                                y - (i * scaleY));

                        g2d.setStroke(hilightStroke);
                        g2d.setColor(Color.GRAY);
                        g2d.draw(line);

                        g2d.setStroke(oldStroke);
                        g2d.setColor(Color.BLACK);
                        g2d.draw(line);
                    }


                } else {
                    // not enough space, use a non-decorated corner
                    float f = 3;
                    Rectangle2D corner = new Rectangle2D.Double(r.getMaxX() - f, r.getMaxY() - f, f, f);
                    g2d.setColor(Color.DARK_GRAY);
                    g2d.fill(corner);
                }

                g2d.setStroke(oldStroke);
                g2d.setColor(orig);
            }

            boolean selected = NodeUtil.isSelected(this, 1);
            if (selected) {
                Color orig = g2d.getColor();
                Stroke origStroke = g2d.getStroke();

                g2d.setColor(Color.RED);
                double s = GeometryUtils.getScale(g2d.getTransform());
                g2d.setStroke(new BasicStroke(1f * s < 1f ? 1f : 1f * (float)s));

                Rectangle2D b = new Rectangle2D.Double(0, 0, width, height);
                Rectangle2D r = b.getBounds2D();
                expand(r, scaleX, scaleY);
                g2d.draw(r);

                g2d.setColor(orig);
                g2d.setStroke(origStroke);
            }

            synchronized(component) {
                component.setLocation((int)g2d.getTransform().getTranslateX(), (int)g2d.getTransform().getTranslateY());
                if(component.getSize().getWidth() != width || component.getSize().getHeight() != height) {
                    component.setSize(width, height);
                }
                component.paint(g2d);
            }

            g2d.setTransform(ot);
        } else {
            /*
             *  Component == null
             *  
             *  The related chart definition ha been removed. 
             */
            System.out.println("TLL, TLL");
        }
    }

    @Override
    protected boolean mouseButtonPressed(MouseButtonPressedEvent event) {
        Point2D local = controlToLocal( event.controlPosition );
        local = parentToLocal(local);
        Rectangle2D bounds = getBoundsInLocal().getBounds2D();
        expand(bounds, 1, 1);
        double control = 3.0;
        Rectangle2D corner = new Rectangle2D.Double(bounds.getMaxX() - control, bounds.getMaxY() - control, control, control);
        if (hover && corner.contains(local)) {
            dragging = true;
            return true;
        }
        return super.mouseButtonPressed(event);
    }


    @Override
    protected boolean mouseMoved(MouseMovedEvent e) {
        if(dragging) {
            Point2D local = controlToLocal( e.controlPosition );
            local = parentToLocal(local);

            Rectangle2D bounds = getBoundsInLocal().getBounds2D();

            if(Math.abs(bounds.getMaxX() - local.getX()) > 3 ||
                    Math.abs(bounds.getMaxY() - local.getY()) > 3) {
                resize(local);
            }
            return true;
        }
        return false;
    }

    public boolean hitCorner(Point2D point) {
        Rectangle2D bounds = getBoundsInLocal().getBounds2D();
        expand(bounds, 1, 1);
        double control = 3.0;
        Rectangle2D corner = new Rectangle2D.Double(bounds.getMaxX() - control, bounds.getMaxY() - control, control, control);
        return corner.contains(point);
    }

    private void resize(Point2D local) {
        Rectangle2D bounds = getBoundsInLocal().getBounds2D();
        double x = bounds.getX();
        double y = bounds.getY();
        double dx = local.getX() - bounds.getMaxX();
        double dy = local.getY() - bounds.getMaxY();
        double w = bounds.getWidth() + dx;
        double h = bounds.getHeight() + dy;
        bounds.setRect(
                x, 
                y, 
                w > 20 ? w : 20, 
                        h > 20 ? h : 20
                );
        setBounds(bounds);
    }

    @Override
    protected boolean mouseDragged(MouseDragBegin e) {
        if(dragging) {
            return true; // Eat event for faster resize
        } else {
            return false;
        }
    }
    
    @Override
    protected boolean mouseButtonReleased(MouseButtonReleasedEvent e) {
        if(dragging) {
            Point2D local = controlToLocal( e.controlPosition );
            local = parentToLocal(local);
            if(Math.abs(bounds.getMaxX() - local.getX()) > 3 ||
                    Math.abs(bounds.getMaxY() - local.getY()) > 3) {
                resize(local);
            }
            dragging = false;
            if(resizeListener != null)
                resizeListener.elementResized(bounds);
        }
        return false;
    }

    @Override
    public int getEventMask() {
        return EventTypes.MouseButtonPressedMask
                | EventTypes.MouseMovedMask
                | EventTypes.MouseButtonReleasedMask
                ;
    }
}
