/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart;

import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.plot.Plot;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.renderer.xy.XYItemRenderer;
import org.jfree.chart.ui.RectangleInsets;
import org.jfree.data.xy.XYDataset;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.sysdyn.JFreeChartResource;

/**
 * Class representing a JFreeChart.XYPlot
 * 
 * @author Teemu Lempinen
 *
 */
public class XYPlot extends AbstractPlot {

    public XYPlot(ReadGraph graph, Resource plotResource) {
        super(graph, plotResource);
    }

    @Override
    protected Plot newPlot() {
        return new org.jfree.chart.plot.XYPlot(null, null, null, null);
    }

    @Override
    protected void getOtherProperties(ReadGraph graph, PlotProperties properties) throws DatabaseException {
        JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
        Boolean visibleGrid = graph.getPossibleRelatedValue(resource, jfree.Plot_visibleGrid);
        properties.otherProperties.put("visibleGrid", visibleGrid);
        if(!properties.datasets.isEmpty()) {
            IDataset idataset = properties.datasets.get(0);
            Resource renderer = graph.getPossibleObject(idataset.getResource(), jfree.Dataset_renderer);
            if(renderer != null) {
                properties.otherProperties.put("renderer", graph.adapt(renderer, IRenderer.class));
            }
        }
    }

    @Override
    protected void setPlotProperties(PlotProperties properties) {
        if(!(plot instanceof org.jfree.chart.plot.XYPlot))
            return;

        org.jfree.chart.plot.XYPlot xyplot = (org.jfree.chart.plot.XYPlot)plot;
        xyplot.clearDomainAxes();
        xyplot.clearRangeAxes();

        for(int i = 0; i < properties.ranges.size(); i++) {
            xyplot.setRangeAxis(i, (ValueAxis)properties.ranges.get(i).getAxis());
        }

        for(int i = 0; i < properties.domains.size(); i++) {
            xyplot.setDomainAxis(i, (ValueAxis)properties.domains.get(i).getAxis());
        }

        IAxis axis;
        for(int i = 0; i < properties.datasets.size(); i++) {
            IDataset dataset = properties.datasets.get(i);
            xyplot.setDataset(i, (XYDataset)dataset.getDataset());
            xyplot.setRenderer(i, (XYItemRenderer)dataset.getRenderer());
            axis = properties.rangeMappings.get(dataset);
            if(axis != null && properties.ranges.contains(axis))
                xyplot.mapDatasetToRangeAxis(i, properties.ranges.indexOf(axis));
            axis = properties.domainMappings.get(dataset);
            if(axis != null && properties.ranges.contains(axis))
                xyplot.mapDatasetToDomainAxis(i, properties.domains.indexOf(axis));
        }

        Boolean visibleGrid = (Boolean)properties.otherProperties.get("visibleGrid");
        if(visibleGrid == null)
            visibleGrid = true;
        
        Boolean orientation = (Boolean)properties.otherProperties.get("orientation");
        if(orientation != null) {
        	if (orientation)
        		xyplot.setOrientation(PlotOrientation.HORIZONTAL);
        	else
        		xyplot.setOrientation(PlotOrientation.VERTICAL);
        }

        xyplot.setRangeGridlinesVisible(visibleGrid);
        xyplot.setDomainGridlinesVisible(visibleGrid);

        // Cleaner look: no outline borders
        xyplot.setInsets(new RectangleInsets(2,5,2,10), false);
        xyplot.setOutlineVisible(false);
    }

}
