/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart;

import java.awt.Color;
import java.awt.Font;
import java.util.HashMap;

import org.jfree.chart.labels.StandardPieSectionLabelGenerator;
import org.jfree.chart.labels.StandardPieToolTipGenerator;
import org.jfree.chart.plot.DefaultDrawingSupplier;
import org.jfree.chart.plot.Plot;
import org.jfree.chart.ui.RectangleInsets;
import org.jfree.data.general.Dataset;
import org.jfree.data.general.DatasetChangeEvent;
import org.jfree.data.general.DatasetChangeListener;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.sysdyn.JFreeChartResource;

/**
 * Class representing a PiePlot in JFreeChart ontology
 * 
 * @author Teemu Lempinen
 *
 */
public class PiePlot extends AbstractPlot {

    protected org.jfree.data.general.PieDataset pieDataset;
    private DatasetChangeListener changeListener;
    
    public PiePlot(ReadGraph graph, Resource resource) {
        super(graph, resource);
    }

    /**
     * Pie plot class with a stricter equals condition
     * @author Teemu Lempinen
     *
     */
    protected static class MyPiePlot extends org.jfree.chart.plot.PiePlot {

        private static final long serialVersionUID = -5917620061541212934L;

        @Override
        public boolean equals(Object obj) {
            boolean result = super.equals(obj);
            if(result == true) {
                org.jfree.chart.plot.PiePlot that = (org.jfree.chart.plot.PiePlot) obj;
                if (this.getDataset() != that.getDataset()) {
                    return false; // Normally plot does not check this. We need this to properly update the charts
                }
            }

            return result;
        }

    }

    @Override
    protected Plot newPlot() {
        MyPiePlot plot = new MyPiePlot();
        plot.setToolTipGenerator(new StandardPieToolTipGenerator());
        return plot;
    }
    
    @Override
    public void dispose() {
    	pieDataset = null;
    	changeListener = null;
    	super.dispose();
    }

    @Override
    protected void getOtherProperties(ReadGraph graph, PlotProperties properties) throws DatabaseException {
        Boolean labelsVisible = graph.getPossibleRelatedValue(resource, JFreeChartResource.getInstance(graph).Plot_visibleLabels, Bindings.BOOLEAN);
        properties.otherProperties.put("labelsVisible", labelsVisible);
        
        Boolean useFilter = graph.getPossibleRelatedValue(resource, JFreeChartResource.getInstance(graph).Filter_used, Bindings.BOOLEAN);
        Double  filterFraction = graph.getPossibleRelatedValue(resource, JFreeChartResource.getInstance(graph).Filter_fraction, Bindings.DOUBLE);
        properties.otherProperties.put("useFilter", useFilter);
        properties.otherProperties.put("filterFraction", filterFraction);
    }
    
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @Override
    protected void setPlotProperties(PlotProperties properties) {
        if(!(plot instanceof org.jfree.chart.plot.PiePlot))
            return;
        
        final org.jfree.chart.plot.PiePlot piePlot = (org.jfree.chart.plot.PiePlot)plot;
        
        if(!properties.datasets.isEmpty()) {
            // We assume that a pie plot has only one dataset
            final IDataset ds = properties.datasets.get(0);
            Dataset dataset = ((PieDataset)ds).getDataset();
            
            if(dataset == null)
                return;

            if(pieDataset != null && changeListener != null) {
                pieDataset.removeChangeListener(changeListener);
            }
            
            pieDataset = (org.jfree.data.general.PieDataset)dataset;
            piePlot.setDataset(pieDataset);
            
            if (pieDataset instanceof FilteredDataset) {
            	FilteredDataset f = (FilteredDataset)pieDataset;
            	Boolean useFilter = (Boolean)properties.otherProperties.get("useFilter");
                Double filterFraction = (Double)properties.otherProperties.get("filterFraction");
                if (useFilter != null && filterFraction != null) {
                	f.setFiltering(useFilter);
                	f.setFilterFraction(filterFraction*0.01);
                	f.updateFiltered();
                } else {
                	f.setFiltering(false);
                }
            }
            
            Boolean labelsVisible = (Boolean)properties.otherProperties.get("labelsVisible");
            if(Boolean.FALSE.equals(labelsVisible))
                piePlot.setLabelGenerator(null);
            else if(piePlot.getLabelGenerator() == null)
                piePlot.setLabelGenerator(new StandardPieSectionLabelGenerator());
            
            changeListener = new DatasetChangeListener() {
                
				@Override
                public void datasetChanged(DatasetChangeEvent event) {
                    HashMap<Comparable<?>, Color> colorMap = ((PieDataset)ds).getColorMap();
                    HashMap<Comparable<?>, Boolean> explodedMap = ((PieDataset)ds).getExplodedMap();
                    
                    for(Object o : piePlot.getDataset().getKeys()) {
                        if(o instanceof Comparable) {
                            Comparable<?> key = (Comparable<?>)o;
                            if(explodedMap.containsKey(key) && explodedMap.get(key)) {
                                piePlot.setExplodePercent(key, 0.3);

                            } else {
                                piePlot.setExplodePercent(key, 0);
                            }
                        }
                    }
                    
                    for(Comparable<?> name : explodedMap.keySet()) {
                        Boolean exploded = explodedMap.get(name);
                        if(Boolean.TRUE.equals(exploded))
                            piePlot.setExplodePercent(name, 0.3);
                    } 
                    piePlot.clearSectionPaints(false);
                    piePlot.setDrawingSupplier(new DefaultDrawingSupplier());
                    for(Comparable<?> name : colorMap.keySet())
                        piePlot.setSectionPaint(name, colorMap.get(name));
                }
            };
            
            pieDataset.addChangeListener(changeListener);
        }
        
        // Cleaner look: no outline borders
        piePlot.setInsets(new RectangleInsets(0,0,0,0), false);
        piePlot.setOutlineVisible(false);
        piePlot.setLabelBackgroundPaint(Color.WHITE);
        piePlot.setLabelFont(new Font("helvetica", Font.PLAIN, 11));
        super.setPlotProperties(properties);
    }

}
