/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart;

import java.awt.Color;
import java.awt.Font;
import java.util.Collection;

import javax.swing.SwingUtilities;

import org.jfree.chart.title.LegendTitle;
import org.jfree.chart.title.TextTitle;
import org.jfree.chart.ui.RectangleInsets;
import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.common.request.PossibleObjectWithType;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.Listener;
import org.simantics.db.request.Read;
import org.simantics.layer0.Layer0;
import org.simantics.sysdyn.JFreeChartResource;
import org.simantics.utils.datastructures.Pair;

/**
 * Class representing a complete JFreeChart.Chart
 * 
 * This class supports all types of charts. The details of the chart are 
 * defined in plots and other adapted classes.
 * 
 * @author Teemu Lempinen
 *
 */
public class JFreeChart implements IJFreeChart {

    protected org.jfree.chart.JFreeChart jfreechart;
    protected JFreeChartListener listener;
    protected IPlot plot;
    protected ITitle title;
    protected Resource chartResource;

    /**
     * 
     * @param graph ReadGraph
     * @param chartResource Resource of type JFreeChart.Chart
     */
    public JFreeChart(ReadGraph graph, Resource chartResource) {
        this.chartResource = chartResource; 
        initialize(graph);
    }
    
    protected JFreeChart() {
    	// default constructor is used with inherited classes.
    }
    
    protected void initialize(ReadGraph graph) {
    	 try {
             JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
             Layer0 l0 = Layer0.getInstance(graph);

             Collection<Resource> plotsCollection = graph.syncRequest(new ObjectsWithType(chartResource, l0.ConsistsOf, jfree.Plot));
             for(Resource plotResource : plotsCollection) {
                 this.plot = getPlot(graph,plotResource);
             } 

         } catch(DatabaseException e) {
             e.printStackTrace();
         }
    }
    
    protected IPlot getPlot(ReadGraph graph, Resource plotResource) throws DatabaseException{
    	return graph.adapt(plotResource, IPlot.class);
    }


    

    /**
     * Returns a new chart using the information collected in the constructor
     */
    @Override
    public org.jfree.chart.JFreeChart getChart() {
        if(plot == null)
            return null;

        if(jfreechart == null)
            jfreechart = new org.jfree.chart.JFreeChart(plot.getPlot());
        
        plot.setJFreeChart(jfreechart);
        
        if(listener == null) {
            listener = new JFreeChartListener();
            Simantics.getSession().asyncRequest(new JFreeChartQuery(), listener);
        }

        return jfreechart;
    }
    
    @Override
    public IPlot getPlot() {
    	return plot;
    }

    @Override
    public void dispose() {
        // Call dispose to title and plots to disable their possible listeners
        if(title != null)
            title.dispose();
        if(listener != null)
            listener.dispose();
        if(plot != null)
            plot.dispose();
    }

    protected class JFreeChartQuery implements Read<Pair<ITitle, Boolean>> {
    	
    	public JFreeChartQuery() {
		}

        @Override
        public Pair<ITitle, Boolean> perform(ReadGraph graph) throws DatabaseException {
            if(chartResource == null || !graph.hasStatement(chartResource))
                return null;
            
            JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
            Layer0 l0 = Layer0.getInstance(graph);

            Resource titleResource = graph.syncRequest(new PossibleObjectWithType(chartResource, l0.ConsistsOf, jfree.Title));
            title = graph.adapt(titleResource, ITitle.class);
            Boolean legendVisible = graph.getPossibleRelatedValue(chartResource, jfree.Chart_visibleLegend, Bindings.BOOLEAN);
            return new Pair<ITitle, Boolean>(title, legendVisible);
        }
    }
    
    protected class JFreeChartListener implements Listener<Pair<ITitle, Boolean>> {

        private boolean disposed = false;
        private LegendTitle legend;
        
        public JFreeChartListener() {
        	
        }

        public void dispose() {
            disposed = true;
        }

        @Override
        public void execute(final Pair<ITitle, Boolean> result) {
            if(result == null)
                return;
            
            SwingUtilities.invokeLater(new Runnable() {
                @Override
                public void run() {
                    if(jfreechart == null)
                        return;

                    jfreechart.setBackgroundPaint(Color.WHITE);
                    
                    if(jfreechart.getLegend() != null && !jfreechart.getLegend().equals(legend)) {
                        legend = jfreechart.getLegend(); 
                        legend.setBorder(0, 0, 0, 0);
                        int size = legend.getItemFont().getSize();
                        legend.setItemFont(new Font("helvetica", Font.PLAIN, size));
                    }

                    if(Boolean.FALSE.equals(result.second)) {
                        jfreechart.removeLegend();
                    } else if (jfreechart.getLegend() == null && legend != null){
                        jfreechart.addLegend(legend);
                    }

                    TextTitle  t = (org.jfree.chart.title.TextTitle)result.first.getTitle();
                    if(t.isVisible()) {
                        t.setFont(new Font("georgia", Font.BOLD, 13));
                        t.setPadding(new RectangleInsets(4, 0, 0, 0));
                        jfreechart.setTitle(t);
                    } else {
                        jfreechart.setTitle((TextTitle)null);
                    }
                }
            });
        }

        @Override
        public void exception(Throwable t) {
            t.printStackTrace();
        }

        @Override
        public boolean isDisposed() {
            return disposed;
        }

    }

}
