/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart;

import java.util.HashMap;

import javax.swing.SwingUtilities;

import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.Axis;
import org.jfree.chart.plot.Plot;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.common.utils.ListUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.Listener;
import org.simantics.db.request.Read;
import org.simantics.layer0.Layer0;
import org.simantics.sysdyn.JFreeChartResource;

/**
 * Abstract plot class for all JFreeChart plots
 * 
 * @author Teemu Lempinen
 *
 */
public abstract class AbstractPlot implements IPlot {

    protected Resource resource;
    protected Plot plot;
    protected PlotProperties currentProperties;
    protected PlotPropertyListener listener;
    protected org.jfree.chart.JFreeChart jfreechart;

    public AbstractPlot(ReadGraph graph, Resource resource) {
        this.resource = resource;
    }

    @Override
    public void dispose() {
        if(currentProperties != null) {
            for(IAxis axis : currentProperties.ranges)
                axis.dispose();

            for(IAxis axis : currentProperties.domains)
                axis.dispose();

            for(IDataset dataset : currentProperties.datasets)
                dataset.dispose();
            currentProperties = null;
        }
        if(listener != null) {
            listener.dispose();
            listener = null;
        }
        jfreechart = null;
        plot = null;
    }

    @Override
    public Resource getResource() {
        return resource;
    }
    
    @Override
    public void setJFreeChart(JFreeChart jfreechart) {
    	this.jfreechart = jfreechart;
    }
    
    public org.jfree.chart.JFreeChart getJfreechart() {
		return jfreechart;
	}
    
    public PlotProperties getPlotProperties() {
    	return currentProperties;
    }

    protected abstract Plot newPlot();
    protected void setPlotProperties(PlotProperties properties) {
        if (currentProperties != null) {
            for (IAxis axis : currentProperties.ranges)
                if (!properties.ranges.contains(axis))
                    axis.dispose();

            for (IAxis axis : currentProperties.domains)
                if (!properties.domains.contains(axis))
                    axis.dispose();

            for (IDataset dataset : currentProperties.datasets)
                if (!properties.datasets.contains(dataset))
                    dataset.dispose();
        }
        this.currentProperties = properties;
    }
    protected abstract void getOtherProperties(ReadGraph graph, PlotProperties properties) throws DatabaseException;

    @Override
    public Plot getPlot() {
        if(plot == null)
            plot = newPlot();

        if(listener == null || listener.isDisposed()) {
            listener = new PlotPropertyListener();
            Simantics.getSession().asyncRequest(new PlotPropertyQuery(), listener);
        }

        return plot;
    }
    
    protected IDataset getDataset(ReadGraph graph, Resource datasetResource) throws DatabaseException{
    	  return graph.adapt(datasetResource, IDataset.class);
    }
    
    @Override
    public void configurePlot(PlotProperties properties) {
    	
    }

    protected class PlotPropertyQuery implements Read<PlotProperties> {
    	
    	public PlotPropertyQuery() {
    		
		}

        @Override
        public PlotProperties perform(ReadGraph graph) throws DatabaseException {

            PlotProperties properties = new PlotProperties();

            Layer0 l0 = Layer0.getInstance(graph);
            JFreeChartResource jfree = JFreeChartResource.getInstance(graph);

            HashMap<Resource, IAxis> axisMap = new HashMap<Resource, IAxis>();

            // Get all range axis
            Resource rangeList = graph.getPossibleObject(resource, jfree.Plot_rangeAxisList);
            if(rangeList != null) {
                for(Resource axisResource : ListUtils.toList(graph, rangeList)) {
                    IAxis axis = graph.adapt(axisResource, IAxis.class);
                    if(axis.getAxis() instanceof Axis) {
                        properties.ranges.add(axis);
                        axisMap.put(axisResource, axis);
                    }
                }
            }

            // Get all domain axis
            // There usually is only one domain axis, but this supports also multiple domain axis
            for(Resource axisResource : graph.syncRequest(new ObjectsWithType(resource, jfree.Plot_domainAxis, jfree.Axis))) {
                IAxis axis = graph.adapt(axisResource, IAxis.class);
                if(axis.getAxis() instanceof Axis) {
                    properties.domains.add(axis);
                    axisMap.put(axisResource, axis);
                }
            }

            // Get all datasets and map them to axis
            for(Resource datasetResource : graph.syncRequest(new ObjectsWithType(resource, l0.ConsistsOf, jfree.Dataset))) {
            	IDataset dataset = getDataset(graph, datasetResource);
                if(dataset != null) {
                    properties.datasets.add(dataset);
                    Resource axisResource = graph.getPossibleObject(datasetResource, jfree.Dataset_mapToRangeAxis);
                    IAxis axis;
                    if(axisMap.containsKey(axisResource)) {
                        axis = axisMap.get(axisResource);
                        properties.rangeMappings.put(dataset, axis);
                    }

                    axisResource = graph.getPossibleObject(datasetResource, jfree.Dataset_mapToDomainAxis);
                    if(axisMap.containsKey(axisResource)) {
                        axis = axisMap.get(axisResource);
                        properties.domainMappings.put(dataset, axis);
                    }
                }
            }
            getOtherProperties(graph, properties);
            return properties;

        }
    }
    
    protected class PlotPropertyListener implements Listener<PlotProperties> {

        private boolean disposed = false;
        
        public PlotPropertyListener() {
        	
        }

        public void dispose() {
            disposed = true;
        }
        @Override
        public void execute(final PlotProperties result) {
            SwingUtilities.invokeLater(new Runnable() {

                @Override
                public void run() {
                    setPlotProperties(result);
                    configurePlot(result);
                }
            });
        }

        @Override
        public void exception(Throwable t) {
            t.printStackTrace();
        }

        @Override
        public boolean isDisposed() {
            return disposed;
        }

    }
}
