/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart;

import java.awt.Color;

import org.jfree.chart.axis.Axis;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.G2DUtils;
import org.simantics.layer0.Layer0;
import org.simantics.sysdyn.JFreeChartResource;

/**
 * Abstract axis class for all JFreeChart axis
 * @author Teemu Lempinen
 *
 */
public abstract class AbstractAxis implements IAxis {

    protected Axis axis;
    protected String label;
    protected Boolean tMarksVisible, tLabelsVisible, labelVisible, lineVisible;
    protected Color color;
    protected Double min, max, rotate;

    /**
     * Creates a new axis
     * @param graph ReadGraph
     * @param axisResource resource of type JFreeChart.NumberAxis
     */
    public AbstractAxis(ReadGraph graph, Resource axisResource) {
        try {
            /*
             *  Axis is practically always called from a listener, 
             *  so it is safe to always create a new axis every time. 
             *  
             *  The parent listener takes care that the axis is updated. 
             *  (And the code stays much more readable)
             */
            Layer0 l0 = Layer0.getInstance(graph);
            JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
            label = graph.getPossibleRelatedValue(axisResource, l0.HasLabel);
            tMarksVisible = graph.getPossibleRelatedValue(axisResource, jfree.Axis_visibleTickMarks, Bindings.BOOLEAN);
            tLabelsVisible = graph.getPossibleRelatedValue(axisResource, jfree.Axis_visibleTickLabels, Bindings.BOOLEAN);
            labelVisible = graph.getPossibleRelatedValue(axisResource, jfree.Axis_visibleLabel, Bindings.BOOLEAN);
            lineVisible = graph.getPossibleRelatedValue(axisResource, jfree.Axis_visibleAxisLine, Bindings.BOOLEAN);
            Resource c = graph.getPossibleObject(axisResource, jfree.color);
            color = c == null ? null : G2DUtils.getColor(graph, c);
            min = graph.getPossibleRelatedValue(axisResource, jfree.Axis_min, Bindings.DOUBLE);
            max = graph.getPossibleRelatedValue(axisResource, jfree.Axis_max, Bindings.DOUBLE);
            rotate = graph.getPossibleRelatedValue(axisResource, jfree.Axis_rotateLabelDegrees, Bindings.DOUBLE);
        } catch (DatabaseException e) {
            e.printStackTrace();
        }
    }
    
    @Override
    public void dispose() {
        
    }

    @Override
    public Axis getAxis() {
        if(tMarksVisible != null && tMarksVisible == false)
            axis.setTickMarksVisible(false);
        if(tLabelsVisible != null && tLabelsVisible == false)
            axis.setTickLabelsVisible(false);
        if(lineVisible != null && lineVisible == false)
            axis.setAxisLineVisible(false);
        
        if(color != null) {
            axis.setAxisLinePaint(color);
            axis.setLabelPaint(color);
            axis.setTickLabelPaint(color);
            axis.setTickMarkPaint(color);
        }
        // label exists and its visibility == null or true
        if((labelVisible == null || labelVisible == true) && label != null)
            axis.setLabel(label);
        return axis;
    }
}
