/*******************************************************************************
 * Copyright (c) 2026 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.issues.ui.handler;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExecutableExtension;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ContributionItem;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.resource.LocalResourceManager;
import org.eclipse.jface.resource.ResourceManager;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.ui.ISelectionService;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.BinaryRead;
import org.simantics.db.common.utils.TagUtil;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.function.DbConsumer;
import org.simantics.db.procedure.Procedure;
import org.simantics.issues.ontology.IssueResource;
import org.simantics.modeling.ModelingResources;
import org.simantics.ui.selection.WorkbenchSelectionUtils;
import org.simantics.utils.ui.BundleUtils;
import org.simantics.utils.ui.ExceptionUtils;
import org.simantics.utils.ui.SWTUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Tuukka Lehtonen
 * @since 1.67.0
 */
public class ResolveRelatedIssuesContribution extends ContributionItem implements IExecutableExtension {

	private static final Logger LOGGER = LoggerFactory.getLogger(ResolveRelatedIssuesContribution.class);

	protected ResourceManager resourceManager;
	protected ImageDescriptor tickIcon;
	protected ImageDescriptor bulbIcon;
	protected boolean resolve = true;
	protected boolean unresolve = false;

	/**
	 * Used to ensure that already old menu populations are not populated at all.
	 */
	protected int modCount = 0;

	public ResolveRelatedIssuesContribution() {
		resourceManager = new LocalResourceManager(JFaceResources.getResources());
		tickIcon = BundleUtils.getImageDescriptorFromPlugin("com.famfamfam.silk", "icons/tick.png");
		bulbIcon = BundleUtils.getImageDescriptorFromPlugin("com.famfamfam.silk", "icons/lightbulb_off.png");
	}

	@Override
	public void setInitializationData(IConfigurationElement config, String propertyName, Object data) throws CoreException {
		if ("class".equals(propertyName)) {
			if (data instanceof String params) {
				String[] parameters = params.split(";"); //$NON-NLS-1$
				for (String parameter : parameters) {
					String[] keyValue = parameter.split("="); //$NON-NLS-1$
					if (keyValue.length == 2) {
						if ("resolve".equals(keyValue[0])) {
							resolve = Boolean.parseBoolean(keyValue[1]);
						}
						if ("unresolve".equals(keyValue[0])) {
							unresolve = Boolean.parseBoolean(keyValue[1]);
						}
					}
				}
			}
		}
	}

	@Override
	public void dispose() {
		if (resourceManager != null) {
			resourceManager.dispose();
			resourceManager = null;
		}
		super.dispose();
	}

	@Override
	public boolean isDynamic() {
		return true;
	}

	protected IStructuredSelection getSelection() {
		IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		ISelectionService service = window.getSelectionService();
		ISelection sel = service.getSelection();
		return sel instanceof IStructuredSelection ss ? ss : null;
	}

	@Override
	public void fill(Menu menu, int index) {
		if (!resolve && !unresolve)
			return;

		var selection = getSelection();
		if (selection.isEmpty())
			return;

		int myCount = ++modCount;
		var display = menu.getDisplay();

		Simantics.getSession().asyncRequest(
				new FindRelatedIssues(selection, resolve),
				new Procedure<>() {
					@Override
					public void execute(List<IssueState> relatedIssues) {
						if (relatedIssues.isEmpty())
							return;
						SWTUtils.asyncExec(display,
								() -> contributeMenuItems(menu, index, relatedIssues, myCount));
					}
					@Override
					public void exception(Throwable t) {
						LOGGER.error("Discovery of resolvable related issues for menu failed.", t); //$NON-NLS-1$
					}
				});
	}

	private void contributeMenuItems(Menu parentMenu, int index, List<IssueState> relatedIssues, int myCount) {
		if (relatedIssues.isEmpty()
				|| myCount != modCount
				|| parentMenu.isDisposed()
				|| resourceManager == null)
			return;

		if (unresolve) {
			var issues = relatedIssues.stream().filter(IssueState::resolved).map(IssueState::issue).toList();
			if (!issues.isEmpty()) {
				addMenuItem(parentMenu, index, new ResolveIssues("Unresolve Related Issues", bulbIcon, false, issues));
			}
		}
		if (resolve) {
			var issues = relatedIssues.stream().filter(IssueState::notResolved).map(IssueState::issue).toList();
			if (!issues.isEmpty()) {
				addMenuItem(parentMenu, index, new ResolveIssues("Resolve Related Issues", tickIcon, true, issues));
			}
		}
	}

	private record IssueState(Resource issue, boolean resolved) {
		boolean notResolved() { return !resolved(); }
	}

	protected static class FindRelatedIssues extends BinaryRead<IStructuredSelection, Boolean, List<IssueState>> {
		public FindRelatedIssues(IStructuredSelection s, boolean resolved) {
			super(s, resolved);
		}

		@Override
		public List<IssueState> perform(ReadGraph graph) throws DatabaseException {
			LOGGER.trace(FindRelatedIssues.class.getName());
			return findRelatedIssues(graph, parameter, parameter2);
		}
	}

	private static List<IssueState> findRelatedIssues(ReadGraph graph, IStructuredSelection s, boolean resolved) throws DatabaseException {
		var wses = WorkbenchSelectionUtils.getWorkbenchSelectionElements(s);
		var issues = new HashSet<IssueState>();

		var ISSUE = IssueResource.getInstance(graph);
		var MOD = ModelingResources.getInstance(graph);

		DbConsumer<Resource> fi = r -> {
			for (var issue : graph.getObjects(r, ISSUE.Issue_HasContext_Inverse)) {
				issues.add(new IssueState(issue, graph.hasStatement(issue, ISSUE.Resolved)));
			}
		};

		for (var wse : wses) {
			var r = WorkbenchSelectionUtils.getPossibleResource(graph, wse);
			if (r != null) {
				fi.accept(r);
				for (Resource component : graph.getObjects(r, MOD.ElementToComponent)) {
					fi.accept(component);
				}
				for (Resource element : graph.getObjects(r, MOD.ComponentToElement)) {
					fi.accept(element);
				}
			}
		}
		return new ArrayList<>(issues);
	}

	private void addMenuItem(Menu parentMenu, int index, ResolveIssues action) {
		MenuItem item = new MenuItem(parentMenu, SWT.PUSH, index);
		item.setText(action.getText());
		item.setToolTipText(action.getToolTipText());
		var img = action.getImageDescriptor();
		if (img != null) {
			item.setImage((Image) resourceManager.get(img));
		}
		item.addSelectionListener(action);
	}

	protected static class ResolveIssues extends Action implements SelectionListener {
		private final boolean resolve;
		private final List<Resource> issues;

		public ResolveIssues(String text, ImageDescriptor image, boolean resolve, List<Resource> issues) {
			super(text, image);
			this.resolve = resolve;
			this.issues = issues;
		}

		@Override
		public void widgetDefaultSelected(SelectionEvent e) {
			widgetSelected(e);
		}

		@Override
		public void widgetSelected(SelectionEvent e) {
			new TagUtil(null, IssueResource.URIs.Resolved, resolve) {
				protected void handleError(DatabaseException ex) {
					var msg = NLS.bind("Failed to {0} for {1} issues", getText(), issues.size());
					LOGGER.error(msg, e);
					ExceptionUtils.logAndShowError("Resolve Failed", msg, ex);
				}
			}.execute(Simantics.getSession(), issues);
		}
	}

}
