/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - GitLab #1148
 *******************************************************************************/
package org.simantics.issues.ui.handler;

import java.lang.reflect.InvocationTargetException;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.osgi.util.NLS;
import org.eclipse.ui.PlatformUI;
import org.simantics.Simantics;
import org.simantics.browsing.ui.common.ErrorLogger;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.DelayedWriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.db.layer0.util.RemoverUtil;
import org.simantics.issues.common.AllActiveIssues;
import org.simantics.issues.common.BatchValidations;
import org.simantics.issues.ontology.IssueResource;

import gnu.trove.map.hash.THashMap;

/**
 * @author Tuukka Lehtonen
 */
public class PurgeResolvedIssues extends AbstractHandler {

    @Override
    public Object execute(ExecutionEvent event) throws ExecutionException {
        try {
            PlatformUI.getWorkbench().getProgressService().busyCursorWhile(new IRunnableWithProgress() {
                @Override
                public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
                    try {
                        purgeResolvedIssues(monitor);
                    } catch (DatabaseException e) {
                        throw new InvocationTargetException(e);
                    } finally {
                        monitor.done();
                    }
                }
            });
        } catch (InvocationTargetException e) {
            ErrorLogger.defaultLogError(e.getCause());
        } catch (InterruptedException e) {
            ErrorLogger.defaultLogError(e);
        }
        return null;
    }

    private void purgeResolvedIssues(IProgressMonitor monitor) throws DatabaseException {
        Session session = Simantics.getSession();
        final Resource project = Simantics.getProjectResource();
        if (project == null)
            return;

        final SubMonitor mon = SubMonitor.convert(monitor, Messages.PurgeResolvedIssues_MonitorPurgingResolvedIssues, 100);

        session.syncRequest(new DelayedWriteRequest() {
            @Override
            public void perform(WriteGraph graph) throws DatabaseException {
                graph.markUndoPoint();
                IssueResource ISSUE = IssueResource.getInstance(graph);
                Set<Resource> toBeRemoved = new HashSet<>();
                Map<Resource, Boolean> sourceIsContinuous = new THashMap<>();
                for (Resource activeIssue : graph.syncRequest(new AllActiveIssues(project))) {
                    if (graph.hasStatement(activeIssue, ISSUE.Resolved)) {
                        Resource managedBy = graph.getPossibleObject(activeIssue, ISSUE.IssueSource_Manages_Inverse);
                        if (managedBy != null) {
                            Boolean isContinuous = sourceIsContinuous.get(managedBy);
                            if (isContinuous == null) {
                                isContinuous = graph.isInstanceOf(managedBy, ISSUE.ContinuousIssueSource);
                                sourceIsContinuous.put(managedBy, isContinuous);
                            }

                            if (isContinuous) {
                                boolean orphanIssue = true;
                                Collection<Resource> contexts = graph.getObjects(activeIssue, ISSUE.Issue_HasContext);
                                if (contexts.size() > 0) {
                                    for (var ctx : contexts) {
                                        if (BatchValidations.isLinkedToOtherThan(graph, ctx, activeIssue)) {
                                            orphanIssue = false;
                                            break;
                                        }
                                    }
                                }
                                if (!orphanIssue)
                                    continue;
                            }
                        }
                        toBeRemoved.add(activeIssue);
                    }
                }

                mon.setTaskName(NLS.bind(Messages.PurgeResolvedIssues_PurgingResolvedBatchIssues, toBeRemoved.size()));
                mon.setWorkRemaining(toBeRemoved.size());
                StringBuilder sb = new StringBuilder();
                sb.append(NLS.bind(Messages.PurgeResolvedIssues_PurgedResolvedBatchIssues, toBeRemoved.size()));
                for (Resource remove : toBeRemoved) {
                    // sb.append(NameUtils.getSafeLabel(graph, remove) + " ");
                    RemoverUtil.remove(graph, remove);
                    mon.worked(1);
                }
                Layer0Utils.addCommentMetadata(graph, sb.toString());
            }
        });
    }

}
