/*******************************************************************************
 * Copyright (c) 2007, 2019 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - Reorganization
 *******************************************************************************/
package org.simantics.issues.ui.handler;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.ICheckStateProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.ui.PlatformUI;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.NamedResource;
import org.simantics.db.common.request.PossibleObjectWithType;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.QueryIndexUtils;
import org.simantics.db.layer0.request.PossibleActiveModel;
import org.simantics.db.layer0.util.RemoverUtil;
import org.simantics.db.request.Write;
import org.simantics.issues.common.IssueUtils;
import org.simantics.issues.ontology.IssueResource;
import org.simantics.layer0.Layer0;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.dialogs.ListDialog;

/**
 * @author Antti Villberg
 */
public class ConfigureIssueSources extends AbstractHandler {

	private static class IssueSourceEntry extends NamedResource {
		private boolean checked;
		public IssueSourceEntry(String name, Resource resource, boolean checked) {
			super(name, resource);
			this.checked = checked;
		}
		public boolean isChecked() {
			return checked;
		}
		public void setChecked(boolean value) {
			checked = value;
		}
	}

	@Override
	public Object execute(ExecutionEvent event) throws ExecutionException {
		try {
			Resource indexRoot = Simantics.getSession().syncRequest(new PossibleActiveModel(Simantics.getProjectResource()));
			executeDefault(indexRoot);
		} catch (DatabaseException e) {
			throw new ExecutionException("Exception while showing validation configuration dialog", e);
		}
		return null;
	}

	public static void executeDefault(Resource indexRoot) throws ExecutionException {

		if (indexRoot == null)
			return;

		try {
			final List<IssueSourceEntry> sources = Simantics.getSession().syncRequest(new UniqueRead<List<IssueSourceEntry>>() {

				@Override
				public List<IssueSourceEntry> perform(ReadGraph graph) throws DatabaseException {

					if(indexRoot == null) return Collections.emptyList();

					List<IssueSourceEntry> result = new ArrayList<IssueSourceEntry>();
					Layer0 L0 = Layer0.getInstance(graph);
					IssueResource ISSUE = IssueResource.getInstance(graph);
					for(Resource type : QueryIndexUtils.searchByType(graph, indexRoot, ISSUE.IssueSourceType)) {
						String name = NameUtils.getSafeLabel(graph, type);
						boolean exists = graph.syncRequest(new PossibleObjectWithType(indexRoot, L0.ConsistsOf, type)) != null;
						boolean deprecated = graph.hasStatement(type, L0.Deprecated);
						boolean abstract_ = graph.hasStatement(type, L0.Abstract);
						if(!exists && (deprecated || abstract_)) continue;
						result.add(new IssueSourceEntry(name, type, exists));
					}
					return result;
				}

			});

			Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
			ListDialog<IssueSourceEntry> dialog = new ListDialog<IssueSourceEntry>(
					shell, sources,
					Messages.ConfigureIssueSources_SelectAvailableIssueSources,
					Messages.ConfigureIssueSources_SelectedSourcesAddRemoveMsg) {

				protected CheckboxTableViewer createViewer(Composite composite) {
					CheckboxTableViewer viewer = CheckboxTableViewer.newCheckList(
							composite, SWT.V_SCROLL | SWT.H_SCROLL | SWT.BORDER);
					viewer.setCheckStateProvider(new ICheckStateProvider() {
						@Override
						public boolean isGrayed(Object o) {
							return false;
						}
						@Override
						public boolean isChecked(Object o) {
							IssueSourceEntry entry = (IssueSourceEntry)o;
							return entry.isChecked();
						}
					});
					viewer.addCheckStateListener(e -> {
						IssueSourceEntry entry = (IssueSourceEntry)e.getElement();
						entry.setChecked(e.getChecked());
					});
					viewer.getTable().addSelectionListener(SelectionListener.widgetSelectedAdapter(e -> {
						viewer.getTable().deselectAll();
					}));
					return viewer;
				}

			};
			int result = dialog.open();
			if (result != Dialog.OK)
				return;

			Simantics.getSession().syncRequest((Write) graph -> {

				Layer0 L0 = Layer0.getInstance(graph);

				for(IssueSourceEntry entry : sources) {

					Resource existing = graph.syncRequest(new PossibleObjectWithType(indexRoot, L0.ConsistsOf, entry.getResource()));

					if(existing == null && entry.isChecked()) {
						String name = NameUtils.getSafeLabel(graph, entry.getResource());
						IssueUtils.addIssueSource(graph, indexRoot, entry.getResource(), name);
					}

					if(existing != null && !entry.isChecked()) {
						RemoverUtil.remove(graph, existing);
					}

				}

			});
		} catch (DatabaseException e) {
			ErrorLogger.defaultLogError(e);
		}

	}

}
