/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.issues.ui.contribution;

import java.util.HashSet;
import java.util.Set;

import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.IDecoration;
import org.simantics.browsing.ui.content.ImageDecorator;
import org.simantics.browsing.ui.model.imagedecorators.ImageDecorationRule;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.issues.Severity;
import org.simantics.issues.common.MaxIssueSeverityRecursive;
import org.simantics.issues.ontology.IssueResource;
import org.simantics.issues.ui.internal.Activator;
import org.simantics.layer0.Layer0;
import org.simantics.utils.ui.gfx.DecorationOverlayIcon;

/**
 * @author Tuukka Lehtonen
 */
public class IssueImageDecorationRule implements ImageDecorationRule {

    Set<Resource> typesToSearch;

    /**
     * @param graph
     * @param typesToSearch URIs of types to recurse into while searching for
     *        issues
     * @throws DatabaseException 
     */
    public IssueImageDecorationRule(ReadGraph graph, String typeToSearch) throws DatabaseException {
        this(graph, new String[] { typeToSearch });
    }

    /**
     * @param graph
     * @param typesToSearch URIs of types to recurse into while searching for
     *        issues
     * @throws DatabaseException 
     */
    public IssueImageDecorationRule(ReadGraph graph, String type1, String type2) throws DatabaseException {
        this(graph, new String[] { type1, type2});
    }

    /**
     * @param graph
     * @param typesToSearch URIs of types to recurse into while searching for
     *        issues
     * @throws DatabaseException 
     */
    public IssueImageDecorationRule(ReadGraph graph, String... typesToSearch) throws DatabaseException {
        this.typesToSearch = new HashSet<Resource>(typesToSearch.length);
        for (String uri : typesToSearch)
            this.typesToSearch.add(graph.getResource(uri));
    }

    @Override
    public boolean isCompatible(Class<?> contentType) {
        return contentType.equals(Resource.class);
    }

    @Override
    public ImageDecorator getImageDecorator(ReadGraph graph, Object content) throws DatabaseException {
        Resource r = (Resource) content;

        // Make sure IssueResource service is initialized.
        Layer0 L0 = Layer0.getInstance(graph);
        IssueResource.getInstance(graph);

        //Severity maxSeverity = graph.syncRequest(new MaxIssueSeveritySingle(r));
        //System.out.println("calculate maxSeverity(" + NameUtils.getSafeName(graph, r, true) + ")");
        Severity maxSeverity = graph.syncRequest(new MaxIssueSeverityRecursive(r, L0.ConsistsOf, typesToSearch));
        //System.out.println("maxSeverity(" + NameUtils.getSafeName(graph, r, true) + "): " + maxSeverity);
        if (maxSeverity == null)
            return null;

        final Severity severity = maxSeverity;
        return new ImageDecorator() {
            @SuppressWarnings("unchecked")
            @Override
            public <Image> Image decorateImage(Image image, String column, int itemIndex) {
                return (Image) getDecoration((ImageDescriptor) image, severity);
            }
        };
    }

    private ImageDescriptor getDecoration(ImageDescriptor original, Severity severity) {
        ImageDescriptor img = Activator.getDefault().getImageRegistry().getDescriptor(severity.toString());
        return (original == null || original.getImageData() == null) ? img
                : new DecorationOverlayIcon(original, img, IDecoration.BOTTOM_LEFT);
    }

}
